'use strict';

/* --------------------------------------------------------------
 slides_container.js 2016-12-29
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Controller Module For Slides Container (Tabs)
 *
 * Handles the sliders container functionality in the sliders details page.
 */
gx.controllers.module('slides_container', [jse.source + '/vendor/jquery-deparam/jquery-deparam.min.js', jse.source + '/vendor/jquery-ui-dist/jquery-ui.min.css', jse.source + '/vendor/jquery-ui-dist/jquery-ui.min.js', 'xhr', 'modal'], function (data) {

    'use strict';

    // --------------------------------------------------------------------
    // VARIABLES
    // --------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);
    var $footer = $('#main-footer');

    /**
     * Elements
     *
     * @type {Object}
     */
    var elements = {
        // Buttons.
        buttons: {
            // Sort mode button.
            sort: $this.find('.sort-button'),

            // Create button.
            create: $this.find('.btn-create'),

            // Submit button group.
            submit: $footer.find('.submit-button-group'),

            // Submit button for save slider
            submitSave: $footer.find('.save'),

            // Submit button for save and refresh slider
            submitRefresh: $footer.find('.refresh')
        },

        // Template.
        templates: {
            // Slide panel set template.
            slidePanel: $this.find('#slide-panel-template')
        },

        // Modals.
        modals: {
            // Delete image modal.
            deleteImage: $('.delete-image.modal'),

            // Delete slide modal.
            deleteSlide: $('.delete-slide.modal'),

            // Edit image map modal.
            imageMap: $('.image-map.modal')
        },

        // Tabs.
        tabHeader: $this.find('.nav-tabs'),

        // Select box which holds all images that will be deleted.
        deleteImageSelect: $('#delete_images')
    };

    /**
     * CSS class names.
     *
     * @type {Object}
     */
    var classes = {
        // New image.
        newImage: 'new'
    };

    /**
     * Selector Strings
     *
     * @type {Object}
     */
    var selectors = {
        // Icon selector strings.
        icons: {
            // Delete button on the panel header.
            delete: '.icon.delete',

            // Drag button on the panel header.
            drag: '.drag-handle',

            // Collapser button on the panel header.
            collapser: '.collapser',

            // Image delete button.
            imageDelete: '.action-icon.delete',

            // Image map edit button.
            imageMap: '.action-icon.image-map',

            // Upload image button.
            upload: '.action-icon.upload'
        },

        // Inputs selector strings.
        inputs: {
            // General image select dropdowns.
            dropdown: '.dropdown-input',

            // Thumbnail dropdown.
            thumbnailImageDropdown: '[name="thumbnail"]',

            // Title.
            title: 'input[name="title"]',

            // File.
            file: '.file-input'
        },

        // Slide panel.
        slidePanel: '.panel',

        // Tab body.
        tabBody: '.tab-pane',

        // Slide panel title.
        slidePanelTitle: '.slide-title',

        // Setting row (form group).
        configurationRow: '.row.form-group',

        // Data list container for image map.
        imageMapDataList: '.image-map-data-list'
    };

    /**
     * Cache list of open slide panels.
     *
     * @type {jQuery[]}
     */
    var openSlidePanels = [];

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // --------------------------------------------------------------------
    // FUNCTIONS
    // --------------------------------------------------------------------

    /**
     * Registers a change, so that the user gets a confirmation dialog while leaving the page.
     */
    function _registerChange() {
        // Object of GET parameters.
        var getParameters = $.deparam(window.location.search.slice(1));

        // Only register in slider edit mode.
        if ('id' in getParameters) {
            window.onbeforeunload = function () {
                return jse.core.lang.translate('EXIT_CONFIRMATION_TEXT', 'sliders');
            };
        }
    }

    /**
     * Handles the image dropdown change event.
     *
     * @param {jQuery.Event} event Triggered event.
     * @param {Boolean} [removeAllDataListItems = false] Remove all data list container list items?
     */
    function _onImageChange(event) {
        var removeAllDataListItems = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;

        // Image dropdown element.
        var $dropdown = $(event.target);

        // Remove icon element.
        var $removeIcon = $dropdown.parents(selectors.configurationRow).find(selectors.icons.imageDelete);

        // Image map icon element.
        var $imageMapIcon = $dropdown.parents(selectors.configurationRow).find(selectors.icons.imageMap);

        // Image map data container list element.
        var $list = $dropdown.parents(selectors.configurationRow).find(selectors.imageMapDataList);

        // Remove the remove icon if 'do not use' is selected.
        $removeIcon[$dropdown.val() ? 'show' : 'hide']();

        // Remove the image map icon if 'do not use' is selected.
        $imageMapIcon[$dropdown.val() ? 'show' : 'hide']();

        // Empty image map data container list.
        $list.children().filter(removeAllDataListItems ? '*' : '.new').remove();
    }

    /**
     * Triggers the file select (click) event of the invisible file input field.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onUploadIconClick(event) {
        $(event.target).parents(selectors.configurationRow).find(selectors.inputs.file).trigger('click');
    }

    /**
     * Handles the file select (change) event.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onImageAdd(event) {
        // Exit method, if selection has been aborted.
        if (!event.target.files.length) {
            return;
        }

        // File input element.
        var $fileInput = $(event.target);

        // Image dropdown.
        var $dropdown = $fileInput.parents(selectors.configurationRow).find(selectors.inputs.dropdown);

        // Regular expression to validate the file name.
        var regex = /(.)(jpg|jpeg|png|gif|bmp)$/i;

        // File name.
        var fileName = event.target.files[0].name;

        // Is the dropdown for thumbnail images?
        var isThumbnailImage = !!$fileInput.parents(selectors.configurationRow).find(selectors.inputs.thumbnailImageDropdown).length;

        // Exit method and show modal, if file type does not match.
        if (!regex.test(fileName)) {
            // Show modal.
            jse.libs.modal.showMessage(jse.core.lang.translate('INVALID_FILE_MODAL_TITLE', 'sliders'), jse.core.lang.translate('INVALID_FILE_MODAL_TEXT', 'sliders'));

            // Reset value.
            $fileInput.val('');

            return;
        }

        // Exit method and show modal, if filename is already present in dropdown.
        var _iteratorNormalCompletion = true;
        var _didIteratorError = false;
        var _iteratorError = undefined;

        try {
            for (var _iterator = $dropdown[0].children[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                var $option = _step.value;

                // Check if option's text content matches with the name of the selected file.
                if ($option.textContent === fileName) {
                    // Show modal.
                    jse.libs.modal.showMessage(jse.core.lang.translate('FILENAME_ALREADY_USED_MODAL_TITLE', 'sliders'), jse.core.lang.translate('FILENAME_ALREADY_USED_MODAL_TEXT', 'sliders'));

                    // Reset value.
                    $fileInput.val('');

                    return;
                }
            }

            // Add files to dropdowns.
        } catch (err) {
            _didIteratorError = true;
            _iteratorError = err;
        } finally {
            try {
                if (!_iteratorNormalCompletion && _iterator.return) {
                    _iterator.return();
                }
            } finally {
                if (_didIteratorError) {
                    throw _iteratorError;
                }
            }
        }

        _addImageToDropdowns(fileName, isThumbnailImage);

        // Select value.
        $dropdown.val(fileName).trigger('change');
    }

    /**
     * Handles the image delete button click event.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onImageDelete(event) {
        // Loading CSS class name.
        var loadingClass = 'loading';

        // Image dropdown container.
        var $configurationRow = $(event.target).parents(selectors.configurationRow);

        // Image dropdown.
        var $dropdown = $configurationRow.find(selectors.inputs.dropdown);

        // Slide ID.
        var slideId = $configurationRow.parents(selectors.slidePanel).data('id');

        // Slide image ID.
        var slideImageId = $configurationRow.data('id');

        // Is the dropdown for thumbnail images?
        var isThumbnailImage = !!$dropdown.is(selectors.inputs.thumbnailImageDropdown);

        // Selected file name.
        var fileName = $dropdown.val();

        // Add loading state.
        $dropdown.addClass(loadingClass);

        // Image usage check request options.
        var requestOptions = {
            url: 'admin.php?do=SlidersDetailsAjax/CheckImageUsage',
            data: {
                filename: fileName,
                is_thumbnail: isThumbnailImage,
                slide_id: slideId,
                slide_image_id: slideImageId
            }
        };

        // Perform deletion.
        var performDeletion = function performDeletion() {
            // Put image name into deleter select box.
            elements.deleteImageSelect.append($('<option>', {
                val: fileName,
                class: isThumbnailImage ? 'thumbnail' : ''
            }));

            // Delete image from dropdowns.
            _deleteImageFromDropdowns(fileName, isThumbnailImage);
        };

        // Check image usage.
        jse.libs.xhr.get(requestOptions).then(function (response) {
            // Remove loading state.
            $dropdown.removeClass(loadingClass);

            if (response.isUsed) {
                // Modal confirmation button element.
                var $confirmButton = elements.modals.deleteImage.find('button.confirm');

                // Show modal.
                elements.modals.deleteImage.modal('show');

                // Listen to confirmation button click event.
                $confirmButton.off('click').on('click', performDeletion);
            } else {
                performDeletion();
            }
        });
    }

    /**
     * Handles the image map edit button click event.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onImageMap(event) {
        // Slide image ID.
        var slideImageId = $(event.target).parents(selectors.configurationRow).data('id');

        // List element which acts like a data container.
        var $list = $(event.target).parents(selectors.configurationRow).find(selectors.imageMapDataList);

        // Image dropdown.
        var $dropdown = $(event.target).parents(selectors.configurationRow).find(selectors.inputs.dropdown);

        // Slide image file name.
        var imageFilename = $dropdown.val();

        // Is a new image selected?
        var isNewImageSelected = $dropdown.find('option:selected').hasClass(classes.newImage);

        // Path to image URL.
        var imageUrl = jse.core.config.get('appUrl') + '/images/slider_images/' + imageFilename;

        // Show save first notice modal and return immediately, if the slide image has no ID.
        if (!slideImageId || isNewImageSelected) {
            jse.libs.modal.showMessage(jse.core.lang.translate('IMAGE_MAP_MODAL_TITLE', 'sliders'), jse.core.lang.translate('SAVE_SLIDER_FIRST_NOTICE_TEXT', 'sliders'));

            return;
        }

        // Show image map modal.
        elements.modals.imageMap.trigger('show', [$list, imageUrl]);
    }

    /**
     * Handles the sort button click event.
     */
    function _onSortButtonClick() {
        // Indicator CSS classes.
        var indicatorClass = 'mode-on btn-primary';

        // Selector string for the slide panel body.
        var slidePanelBodySelector = '.panel-body';

        // Slides container tabs, except the active one.
        var $otherTabs = elements.tabHeader.children().not('.active');

        // Is the sort mode on?
        var isModeOn = elements.buttons.sort.hasClass(indicatorClass);

        // Language-specific button texts.
        var enterText = elements.buttons.sort.data('textEnter');
        var exitText = elements.buttons.sort.data('textExit');

        // All slide panels.
        var $slides = $this.find(selectors.slidePanel);

        // Apply fade effect onto slide panels.
        $slides.hide().fadeIn();

        // Switch text and toggle indicator class.
        elements.buttons.sort[isModeOn ? 'removeClass' : 'addClass'](indicatorClass).text(isModeOn ? enterText : exitText);

        // Toggle create button.
        elements.buttons.create.prop('disabled', !isModeOn);

        // Toggle drag handle buttons.
        $slides.find(selectors.icons.drag)[isModeOn ? 'hide' : 'show']();

        // Toggle other tabs.
        $otherTabs[isModeOn ? 'fadeIn' : 'fadeOut']();

        // Toggle collapser and hide buttons.
        $slides.find(selectors.icons.collapser).add(selectors.icons.delete)[isModeOn ? 'show' : 'hide']();

        // Save open slide panels.
        if (!isModeOn) {
            openSlidePanels = $slides.filter(function (index, element) {
                return $(element).find(slidePanelBodySelector).is(':visible');
            });
        }

        // Toggle saved open slide panels.
        openSlidePanels.each(function (index, element) {
            return $(element).find(selectors.icons.collapser).trigger('click');
        });
    }

    /**
     * Handles the sort start event.
     */
    function _onSortStart() {
        // Tab content element for selected language.
        var $tabBody = $this.find(selectors.tabBody).filter(':visible');

        // Refresh tab sizes and positions.
        $tabBody.sortable('refreshPositions');
    }

    /**
     * Handles the sort stop event.
     */
    function _onSortStop() {
        // Register change, to make prompt on page unload.
        _registerChange();
    }

    /**
     * Handles the delete icon click event.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onDeleteIconClick(event) {
        // Slide panel element.
        var $slidePanel = $(event.target).parents(selectors.slidePanel);

        // Modal confirmation button element.
        var $confirmButton = elements.modals.deleteSlide.find('button.confirm');

        // Show modal.
        elements.modals.deleteSlide.modal('show');

        // Listen to confirmation button click event.
        $confirmButton.off('click').on('click', function () {
            return _onDeleteConfirmationButtonClick(elements.modals.deleteSlide, $slidePanel);
        });
    }

    /**
     * Handles the create button click event.
     */
    function _onCreateButtonClick() {
        // Make a clone of the slide panel template and create a new element.
        var $slidePanel = $(elements.templates.slidePanel.clone().html());

        // Tab content element for selected language.
        var $tabBody = $this.find(selectors.tabBody).filter(':visible');

        // Slide panels.
        var $slidePanels = $tabBody.find(selectors.slidePanel);

        // Next panel index.
        var panelIndex = $slidePanels.length + 1;

        // Title for new slide panel.
        var newTitle = jse.core.lang.translate('NEW_SLIDE', 'sliders') + ' ' + panelIndex;

        // Add title to slide panel header.
        $slidePanel.find(selectors.slidePanelTitle).text(newTitle);

        // Add title to input field.
        $slidePanel.find(selectors.inputs.title).val(newTitle);

        // Add values to dropdowns.
        if ($slidePanels.length) {
            // Get all image dropdowns of the first panel.
            var $dropdowns = $slidePanels.first().find(selectors.inputs.dropdown);

            // Get the thumbnail dropdown options.
            var $thumbnailOptions = $dropdowns.filter(selectors.inputs.thumbnailImageDropdown).children().clone();

            // Get the image dropdown options.
            var $imageOptions = $dropdowns.not(selectors.inputs.thumbnailImageDropdown).first().children().clone();

            // Replace thumbnail options in new slide panel.
            $slidePanel.find(selectors.inputs.thumbnailImageDropdown).empty().append($thumbnailOptions).val('');

            // Replace image options in new slide panel.
            $slidePanel.find(selectors.inputs.dropdown).not(selectors.inputs.thumbnailImageDropdown).empty().append($imageOptions).val('');
        }

        // Add new slide panel element to tab body with fade effect.
        $slidePanel.hide().prependTo($tabBody).fadeIn();

        // Initialize widgets and extensions on the new slide panel element.
        gx.widgets.init($slidePanel);
        gx.extensions.init($slidePanel);

        // Trigger change to show the right action icons.
        $slidePanel.find('select').trigger('change');

        // Register change, to make prompt on page unload.
        _registerChange();

        // Toggle submit buttons.
        toggleSubmitButtons();
    }

    /**
     * Handles the confirmation button click event in the delete confirmation modal.
     *
     * @param {jQuery} $modal Delete confirmation modal element.
     * @param {jQuery} $slidePanel Slide panel element.
     */
    function _onDeleteConfirmationButtonClick($modal, $slidePanel) {
        // Hide modal.
        $modal.modal('hide');

        // Fade out slide panel element and then remove it.
        $slidePanel.fadeOut(400, function () {
            // Remove slide panel.
            $slidePanel.remove();

            // Toggle submit buttons.
            toggleSubmitButtons();
        });

        // Register change, to make prompt on page unload.
        _registerChange();
    }

    /**
     * Handles the key-up event on the slide title input field.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onTitleKeyup(event) {
        // Title input field.
        var $input = $(event.target);

        // Slide panel title element.
        var $title = $input.parents(selectors.slidePanel).find(selectors.slidePanelTitle);

        // Transfer input value to slide panel title.
        $title.text($input.val());
    }

    /**
     * Handles the mouse-enter event on a configuration row.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onConfigRowMouseEnter(event) {
        // Configuration row element.
        var $row = $(event.target);

        // Image map edit icon.
        var $imageMapIcon = $row.find(selectors.icons.imageMap);

        // Image map data container list element.
        var $list = $row.find(selectors.imageMapDataList);

        // Return immediately, if the image map edit icon does not exist.
        if (!$imageMapIcon.length || !$list.length) {
            return;
        }

        if ($list.children().length) {
            $imageMapIcon.removeClass('fa-external-link').addClass('fa-external-link-square');
        } else {
            $imageMapIcon.removeClass('fa-external-link-square').addClass('fa-external-link');
        }
    }

    /**
     * Handles the click event on the save button.
     */
    function _onSubmitSave() {
        $this.parents('form').trigger('submit');
    }

    /**
     * Handles the click event on the refresh list item in the submit button group.
     */
    function _onSubmitRefresh() {
        $this.parents('form').trigger('submit', { refresh: true });
    }

    /**
     * Adds an image to the image dropdowns.
     *
     * @param {String} fileName Name of the selected file.
     * @param {Boolean} [thumbnailImagesOnly = false] Apply on thumbnail image dropdowns only?
     */
    function _addImageToDropdowns(fileName) {
        var thumbnailImagesOnly = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

        // Select specific dropdowns.
        var $dropdowns = $this.find(selectors.inputs.dropdown)[thumbnailImagesOnly ? 'filter' : 'not'](selectors.inputs.thumbnailImageDropdown);

        // Create new image option element.
        var $option = $('<option>', { value: fileName, text: fileName, class: classes.newImage });

        // Append new options to dropdowns.
        $dropdowns.append($option);
    }

    /**
     * Deletes an image from the image dropdowns.
     *
     * @param {String} fileName Name of the selected file.
     * @param {Boolean} [thumbnailImagesOnly = false] Apply on thumbnail image dropdowns only?
     */
    function _deleteImageFromDropdowns(fileName) {
        var thumbnailImagesOnly = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

        // Select all dropdowns.
        var $dropdowns = $this.find(selectors.inputs.dropdown)[thumbnailImagesOnly ? 'filter' : 'not'](selectors.inputs.thumbnailImageDropdown);

        // Remove image option from each dropdown.
        $dropdowns.each(function (index, element) {
            // Dropdown element.
            var $dropdown = $(element);

            // Remove option.
            $dropdown.find('[value="' + fileName + '"]').remove();

            // Set to default value if there are no image file options.
            if ($dropdown.children().length <= 1) {
                $dropdown.val('');
            }

            // Trigger change.
            $dropdown.trigger('change');
        });
    }

    /**
     * Disables or enables the submit buttons.
     */
    function toggleSubmitButtons() {
        // Enable the submit buttons?
        var doEnableSubmitButtons = true;

        // Slides.
        var $slides = $this.find(selectors.slidePanel);

        // Disable submit buttons, if there are no slides.
        if (!$slides.length) {
            doEnableSubmitButtons = false;
        }

        // Disable/Enable submit buttons.
        elements.buttons.submit.children().not('ul').prop('disabled', !doEnableSubmitButtons);
    }

    // --------------------------------------------------------------------
    // INITIALIZATION
    // --------------------------------------------------------------------

    module.init = function (done) {
        // Attach click event handler to sort button.
        elements.buttons.sort.on('click', _onSortButtonClick);

        // Attach event handlers to sort actions, slide panel delete button and inputs fields.
        $this.on('sortstart', _onSortStart).on('sortstop', _onSortStop).on('click', selectors.icons.delete, _onDeleteIconClick).on('keyup', selectors.inputs.title, _onTitleKeyup).on('change', selectors.inputs.file, _onImageAdd).on('click', selectors.icons.upload, _onUploadIconClick).on('click', selectors.icons.imageDelete, _onImageDelete).on('click', selectors.icons.imageMap, _onImageMap).on('change', selectors.inputs.dropdown, _onImageChange).on('mouseenter', selectors.configurationRow, _onConfigRowMouseEnter);

        // Attach event listeners to submit buttons.
        elements.buttons.submitSave.on('click', _onSubmitSave);
        elements.buttons.submitRefresh.on('click', _onSubmitRefresh);

        // Attach click event handler to create button.
        elements.buttons.create.on('click', _onCreateButtonClick);

        // Activate first tab.
        elements.tabHeader.children().first().addClass('active');

        // Activate first tab content.
        $this.find(selectors.tabBody).first().addClass('active in');

        // Trigger dropdown change event to hide the remove icon, if 'do not use' is selected.
        $this.find(selectors.inputs.dropdown).trigger('change', [false]);

        // Finish initialization.
        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInNsaWRlcnMvZGV0YWlscy9zbGlkZXNfY29udGFpbmVyLmpzIl0sIm5hbWVzIjpbImd4IiwiY29udHJvbGxlcnMiLCJtb2R1bGUiLCJqc2UiLCJzb3VyY2UiLCJkYXRhIiwiJHRoaXMiLCIkIiwiJGZvb3RlciIsImVsZW1lbnRzIiwiYnV0dG9ucyIsInNvcnQiLCJmaW5kIiwiY3JlYXRlIiwic3VibWl0Iiwic3VibWl0U2F2ZSIsInN1Ym1pdFJlZnJlc2giLCJ0ZW1wbGF0ZXMiLCJzbGlkZVBhbmVsIiwibW9kYWxzIiwiZGVsZXRlSW1hZ2UiLCJkZWxldGVTbGlkZSIsImltYWdlTWFwIiwidGFiSGVhZGVyIiwiZGVsZXRlSW1hZ2VTZWxlY3QiLCJjbGFzc2VzIiwibmV3SW1hZ2UiLCJzZWxlY3RvcnMiLCJpY29ucyIsImRlbGV0ZSIsImRyYWciLCJjb2xsYXBzZXIiLCJpbWFnZURlbGV0ZSIsInVwbG9hZCIsImlucHV0cyIsImRyb3Bkb3duIiwidGh1bWJuYWlsSW1hZ2VEcm9wZG93biIsInRpdGxlIiwiZmlsZSIsInRhYkJvZHkiLCJzbGlkZVBhbmVsVGl0bGUiLCJjb25maWd1cmF0aW9uUm93IiwiaW1hZ2VNYXBEYXRhTGlzdCIsIm9wZW5TbGlkZVBhbmVscyIsIl9yZWdpc3RlckNoYW5nZSIsImdldFBhcmFtZXRlcnMiLCJkZXBhcmFtIiwid2luZG93IiwibG9jYXRpb24iLCJzZWFyY2giLCJzbGljZSIsIm9uYmVmb3JldW5sb2FkIiwiY29yZSIsImxhbmciLCJ0cmFuc2xhdGUiLCJfb25JbWFnZUNoYW5nZSIsImV2ZW50IiwicmVtb3ZlQWxsRGF0YUxpc3RJdGVtcyIsIiRkcm9wZG93biIsInRhcmdldCIsIiRyZW1vdmVJY29uIiwicGFyZW50cyIsIiRpbWFnZU1hcEljb24iLCIkbGlzdCIsInZhbCIsImNoaWxkcmVuIiwiZmlsdGVyIiwicmVtb3ZlIiwiX29uVXBsb2FkSWNvbkNsaWNrIiwidHJpZ2dlciIsIl9vbkltYWdlQWRkIiwiZmlsZXMiLCJsZW5ndGgiLCIkZmlsZUlucHV0IiwicmVnZXgiLCJmaWxlTmFtZSIsIm5hbWUiLCJpc1RodW1ibmFpbEltYWdlIiwidGVzdCIsImxpYnMiLCJtb2RhbCIsInNob3dNZXNzYWdlIiwiJG9wdGlvbiIsInRleHRDb250ZW50IiwiX2FkZEltYWdlVG9Ecm9wZG93bnMiLCJfb25JbWFnZURlbGV0ZSIsImxvYWRpbmdDbGFzcyIsIiRjb25maWd1cmF0aW9uUm93Iiwic2xpZGVJZCIsInNsaWRlSW1hZ2VJZCIsImlzIiwiYWRkQ2xhc3MiLCJyZXF1ZXN0T3B0aW9ucyIsInVybCIsImZpbGVuYW1lIiwiaXNfdGh1bWJuYWlsIiwic2xpZGVfaWQiLCJzbGlkZV9pbWFnZV9pZCIsInBlcmZvcm1EZWxldGlvbiIsImFwcGVuZCIsImNsYXNzIiwiX2RlbGV0ZUltYWdlRnJvbURyb3Bkb3ducyIsInhociIsImdldCIsInRoZW4iLCJyZW1vdmVDbGFzcyIsInJlc3BvbnNlIiwiaXNVc2VkIiwiJGNvbmZpcm1CdXR0b24iLCJvZmYiLCJvbiIsIl9vbkltYWdlTWFwIiwiaW1hZ2VGaWxlbmFtZSIsImlzTmV3SW1hZ2VTZWxlY3RlZCIsImhhc0NsYXNzIiwiaW1hZ2VVcmwiLCJjb25maWciLCJfb25Tb3J0QnV0dG9uQ2xpY2siLCJpbmRpY2F0b3JDbGFzcyIsInNsaWRlUGFuZWxCb2R5U2VsZWN0b3IiLCIkb3RoZXJUYWJzIiwibm90IiwiaXNNb2RlT24iLCJlbnRlclRleHQiLCJleGl0VGV4dCIsIiRzbGlkZXMiLCJoaWRlIiwiZmFkZUluIiwidGV4dCIsInByb3AiLCJhZGQiLCJpbmRleCIsImVsZW1lbnQiLCJlYWNoIiwiX29uU29ydFN0YXJ0IiwiJHRhYkJvZHkiLCJzb3J0YWJsZSIsIl9vblNvcnRTdG9wIiwiX29uRGVsZXRlSWNvbkNsaWNrIiwiJHNsaWRlUGFuZWwiLCJfb25EZWxldGVDb25maXJtYXRpb25CdXR0b25DbGljayIsIl9vbkNyZWF0ZUJ1dHRvbkNsaWNrIiwiY2xvbmUiLCJodG1sIiwiJHNsaWRlUGFuZWxzIiwicGFuZWxJbmRleCIsIm5ld1RpdGxlIiwiJGRyb3Bkb3ducyIsImZpcnN0IiwiJHRodW1ibmFpbE9wdGlvbnMiLCIkaW1hZ2VPcHRpb25zIiwiZW1wdHkiLCJwcmVwZW5kVG8iLCJ3aWRnZXRzIiwiaW5pdCIsImV4dGVuc2lvbnMiLCJ0b2dnbGVTdWJtaXRCdXR0b25zIiwiJG1vZGFsIiwiZmFkZU91dCIsIl9vblRpdGxlS2V5dXAiLCIkaW5wdXQiLCIkdGl0bGUiLCJfb25Db25maWdSb3dNb3VzZUVudGVyIiwiJHJvdyIsIl9vblN1Ym1pdFNhdmUiLCJfb25TdWJtaXRSZWZyZXNoIiwicmVmcmVzaCIsInRodW1ibmFpbEltYWdlc09ubHkiLCJ2YWx1ZSIsImRvRW5hYmxlU3VibWl0QnV0dG9ucyIsImRvbmUiXSwibWFwcGluZ3MiOiI7O0FBQUE7Ozs7Ozs7Ozs7QUFVQTs7Ozs7QUFLQUEsR0FBR0MsV0FBSCxDQUFlQyxNQUFmLENBQ0ksa0JBREosRUFHSSxDQUNPQyxJQUFJQyxNQURYLG1EQUVPRCxJQUFJQyxNQUZYLCtDQUdPRCxJQUFJQyxNQUhYLDhDQUlJLEtBSkosRUFLSSxPQUxKLENBSEosRUFXSSxVQUFVQyxJQUFWLEVBQWdCOztBQUVaOztBQUVBO0FBQ0E7QUFDQTs7QUFFQTs7Ozs7O0FBS0EsUUFBTUMsUUFBUUMsRUFBRSxJQUFGLENBQWQ7QUFDQSxRQUFNQyxVQUFVRCxFQUFFLGNBQUYsQ0FBaEI7O0FBRUE7Ozs7O0FBS0EsUUFBTUUsV0FBVztBQUNiO0FBQ0FDLGlCQUFTO0FBQ0w7QUFDQUMsa0JBQU1MLE1BQU1NLElBQU4sQ0FBVyxjQUFYLENBRkQ7O0FBSUw7QUFDQUMsb0JBQVFQLE1BQU1NLElBQU4sQ0FBVyxhQUFYLENBTEg7O0FBT0w7QUFDQUUsb0JBQVFOLFFBQVFJLElBQVIsQ0FBYSxzQkFBYixDQVJIOztBQVVMO0FBQ0FHLHdCQUFZUCxRQUFRSSxJQUFSLENBQWEsT0FBYixDQVhQOztBQWFMO0FBQ0FJLDJCQUFlUixRQUFRSSxJQUFSLENBQWEsVUFBYjtBQWRWLFNBRkk7O0FBbUJiO0FBQ0FLLG1CQUFXO0FBQ1A7QUFDQUMsd0JBQVlaLE1BQU1NLElBQU4sQ0FBVyx1QkFBWDtBQUZMLFNBcEJFOztBQXlCYjtBQUNBTyxnQkFBUTtBQUNKO0FBQ0FDLHlCQUFhYixFQUFFLHFCQUFGLENBRlQ7O0FBSUo7QUFDQWMseUJBQWFkLEVBQUUscUJBQUYsQ0FMVDs7QUFPSjtBQUNBZSxzQkFBVWYsRUFBRSxrQkFBRjtBQVJOLFNBMUJLOztBQXFDYjtBQUNBZ0IsbUJBQVdqQixNQUFNTSxJQUFOLENBQVcsV0FBWCxDQXRDRTs7QUF3Q2I7QUFDQVksMkJBQW1CakIsRUFBRSxnQkFBRjtBQXpDTixLQUFqQjs7QUE0Q0E7Ozs7O0FBS0EsUUFBTWtCLFVBQVU7QUFDWjtBQUNBQyxrQkFBVTtBQUZFLEtBQWhCOztBQUtBOzs7OztBQUtBLFFBQU1DLFlBQVk7QUFDZDtBQUNBQyxlQUFPO0FBQ0g7QUFDQUMsb0JBQVEsY0FGTDs7QUFJSDtBQUNBQyxrQkFBTSxjQUxIOztBQU9IO0FBQ0FDLHVCQUFXLFlBUlI7O0FBVUg7QUFDQUMseUJBQWEscUJBWFY7O0FBYUg7QUFDQVYsc0JBQVUsd0JBZFA7O0FBZ0JIO0FBQ0FXLG9CQUFRO0FBakJMLFNBRk87O0FBc0JkO0FBQ0FDLGdCQUFRO0FBQ0o7QUFDQUMsc0JBQVUsaUJBRk47O0FBSUo7QUFDQUMsb0NBQXdCLG9CQUxwQjs7QUFPSjtBQUNBQyxtQkFBTyxxQkFSSDs7QUFVSjtBQUNBQyxrQkFBTTtBQVhGLFNBdkJNOztBQXFDZDtBQUNBcEIsb0JBQVksUUF0Q0U7O0FBd0NkO0FBQ0FxQixpQkFBUyxXQXpDSzs7QUEyQ2Q7QUFDQUMseUJBQWlCLGNBNUNIOztBQThDZDtBQUNBQywwQkFBa0IsaUJBL0NKOztBQWlEZDtBQUNBQywwQkFBa0I7QUFsREosS0FBbEI7O0FBcURBOzs7OztBQUtBLFFBQUlDLGtCQUFrQixFQUF0Qjs7QUFFQTs7Ozs7QUFLQSxRQUFNekMsU0FBUyxFQUFmOztBQUVBO0FBQ0E7QUFDQTs7QUFFQTs7O0FBR0EsYUFBUzBDLGVBQVQsR0FBMkI7QUFDdkI7QUFDQSxZQUFNQyxnQkFBZ0J0QyxFQUFFdUMsT0FBRixDQUFVQyxPQUFPQyxRQUFQLENBQWdCQyxNQUFoQixDQUF1QkMsS0FBdkIsQ0FBNkIsQ0FBN0IsQ0FBVixDQUF0Qjs7QUFFQTtBQUNBLFlBQUksUUFBUUwsYUFBWixFQUEyQjtBQUN2QkUsbUJBQU9JLGNBQVAsR0FBd0I7QUFBQSx1QkFBTWhELElBQUlpRCxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3Qix3QkFBeEIsRUFBa0QsU0FBbEQsQ0FBTjtBQUFBLGFBQXhCO0FBQ0g7QUFDSjs7QUFFRDs7Ozs7O0FBTUEsYUFBU0MsY0FBVCxDQUF3QkMsS0FBeEIsRUFBOEQ7QUFBQSxZQUEvQkMsc0JBQStCLHVFQUFOLElBQU07O0FBQzFEO0FBQ0EsWUFBTUMsWUFBWW5ELEVBQUVpRCxNQUFNRyxNQUFSLENBQWxCOztBQUVBO0FBQ0EsWUFBTUMsY0FBY0YsVUFDZkcsT0FEZSxDQUNQbEMsVUFBVWMsZ0JBREgsRUFFZjdCLElBRmUsQ0FFVmUsVUFBVUMsS0FBVixDQUFnQkksV0FGTixDQUFwQjs7QUFJQTtBQUNBLFlBQU04QixnQkFBZ0JKLFVBQ2pCRyxPQURpQixDQUNUbEMsVUFBVWMsZ0JBREQsRUFFakI3QixJQUZpQixDQUVaZSxVQUFVQyxLQUFWLENBQWdCTixRQUZKLENBQXRCOztBQUlBO0FBQ0EsWUFBTXlDLFFBQVFMLFVBQ1RHLE9BRFMsQ0FDRGxDLFVBQVVjLGdCQURULEVBRVQ3QixJQUZTLENBRUplLFVBQVVlLGdCQUZOLENBQWQ7O0FBSUE7QUFDQWtCLG9CQUFZRixVQUFVTSxHQUFWLEtBQWtCLE1BQWxCLEdBQTJCLE1BQXZDOztBQUVBO0FBQ0FGLHNCQUFjSixVQUFVTSxHQUFWLEtBQWtCLE1BQWxCLEdBQTJCLE1BQXpDOztBQUVBO0FBQ0FELGNBQ0tFLFFBREwsR0FFS0MsTUFGTCxDQUVZVCx5QkFBeUIsR0FBekIsR0FBK0IsTUFGM0MsRUFHS1UsTUFITDtBQUlIOztBQUVEOzs7OztBQUtBLGFBQVNDLGtCQUFULENBQTRCWixLQUE1QixFQUFtQztBQUMvQmpELFVBQUVpRCxNQUFNRyxNQUFSLEVBQ0tFLE9BREwsQ0FDYWxDLFVBQVVjLGdCQUR2QixFQUVLN0IsSUFGTCxDQUVVZSxVQUFVTyxNQUFWLENBQWlCSSxJQUYzQixFQUdLK0IsT0FITCxDQUdhLE9BSGI7QUFJSDs7QUFFRDs7Ozs7QUFLQSxhQUFTQyxXQUFULENBQXFCZCxLQUFyQixFQUE0QjtBQUN4QjtBQUNBLFlBQUksQ0FBQ0EsTUFBTUcsTUFBTixDQUFhWSxLQUFiLENBQW1CQyxNQUF4QixFQUFnQztBQUM1QjtBQUNIOztBQUVEO0FBQ0EsWUFBTUMsYUFBYWxFLEVBQUVpRCxNQUFNRyxNQUFSLENBQW5COztBQUVBO0FBQ0EsWUFBTUQsWUFBWWUsV0FDYlosT0FEYSxDQUNMbEMsVUFBVWMsZ0JBREwsRUFFYjdCLElBRmEsQ0FFUmUsVUFBVU8sTUFBVixDQUFpQkMsUUFGVCxDQUFsQjs7QUFJQTtBQUNBLFlBQU11QyxRQUFRLDZCQUFkOztBQUVBO0FBQ0EsWUFBTUMsV0FBV25CLE1BQU1HLE1BQU4sQ0FBYVksS0FBYixDQUFtQixDQUFuQixFQUFzQkssSUFBdkM7O0FBRUE7QUFDQSxZQUFNQyxtQkFBbUIsQ0FBQyxDQUFDSixXQUN0QlosT0FEc0IsQ0FDZGxDLFVBQVVjLGdCQURJLEVBRXRCN0IsSUFGc0IsQ0FFakJlLFVBQVVPLE1BQVYsQ0FBaUJFLHNCQUZBLEVBR3RCb0MsTUFITDs7QUFLQTtBQUNBLFlBQUksQ0FBQ0UsTUFBTUksSUFBTixDQUFXSCxRQUFYLENBQUwsRUFBMkI7QUFDdkI7QUFDQXhFLGdCQUFJNEUsSUFBSixDQUFTQyxLQUFULENBQWVDLFdBQWYsQ0FDSTlFLElBQUlpRCxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QiwwQkFBeEIsRUFBb0QsU0FBcEQsQ0FESixFQUVJbkQsSUFBSWlELElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQXdCLHlCQUF4QixFQUFtRCxTQUFuRCxDQUZKOztBQUtBO0FBQ0FtQix1QkFBV1QsR0FBWCxDQUFlLEVBQWY7O0FBRUE7QUFDSDs7QUFFRDtBQXhDd0I7QUFBQTtBQUFBOztBQUFBO0FBeUN4QixpQ0FBc0JOLFVBQVUsQ0FBVixFQUFhTyxRQUFuQyw4SEFBNkM7QUFBQSxvQkFBbENpQixPQUFrQzs7QUFDekM7QUFDQSxvQkFBSUEsUUFBUUMsV0FBUixLQUF3QlIsUUFBNUIsRUFBc0M7QUFDbEM7QUFDQXhFLHdCQUFJNEUsSUFBSixDQUFTQyxLQUFULENBQWVDLFdBQWYsQ0FDSTlFLElBQUlpRCxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixtQ0FBeEIsRUFBNkQsU0FBN0QsQ0FESixFQUVJbkQsSUFBSWlELElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQXdCLGtDQUF4QixFQUE0RCxTQUE1RCxDQUZKOztBQUtBO0FBQ0FtQiwrQkFBV1QsR0FBWCxDQUFlLEVBQWY7O0FBRUE7QUFDSDtBQUNKOztBQUVEO0FBekR3QjtBQUFBO0FBQUE7QUFBQTtBQUFBO0FBQUE7QUFBQTtBQUFBO0FBQUE7QUFBQTtBQUFBO0FBQUE7QUFBQTtBQUFBOztBQTBEeEJvQiw2QkFBcUJULFFBQXJCLEVBQStCRSxnQkFBL0I7O0FBRUE7QUFDQW5CLGtCQUNLTSxHQURMLENBQ1NXLFFBRFQsRUFFS04sT0FGTCxDQUVhLFFBRmI7QUFHSDs7QUFFRDs7Ozs7QUFLQSxhQUFTZ0IsY0FBVCxDQUF3QjdCLEtBQXhCLEVBQStCO0FBQzNCO0FBQ0EsWUFBTThCLGVBQWUsU0FBckI7O0FBRUE7QUFDQSxZQUFNQyxvQkFBb0JoRixFQUFFaUQsTUFBTUcsTUFBUixFQUFnQkUsT0FBaEIsQ0FBd0JsQyxVQUFVYyxnQkFBbEMsQ0FBMUI7O0FBRUE7QUFDQSxZQUFNaUIsWUFBWTZCLGtCQUFrQjNFLElBQWxCLENBQXVCZSxVQUFVTyxNQUFWLENBQWlCQyxRQUF4QyxDQUFsQjs7QUFFQTtBQUNBLFlBQU1xRCxVQUFVRCxrQkFBa0IxQixPQUFsQixDQUEwQmxDLFVBQVVULFVBQXBDLEVBQWdEYixJQUFoRCxDQUFxRCxJQUFyRCxDQUFoQjs7QUFFQTtBQUNBLFlBQU1vRixlQUFlRixrQkFBa0JsRixJQUFsQixDQUF1QixJQUF2QixDQUFyQjs7QUFFQTtBQUNBLFlBQU13RSxtQkFBbUIsQ0FBQyxDQUFDbkIsVUFBVWdDLEVBQVYsQ0FBYS9ELFVBQVVPLE1BQVYsQ0FBaUJFLHNCQUE5QixDQUEzQjs7QUFFQTtBQUNBLFlBQU11QyxXQUFXakIsVUFBVU0sR0FBVixFQUFqQjs7QUFFQTtBQUNBTixrQkFBVWlDLFFBQVYsQ0FBbUJMLFlBQW5COztBQUVBO0FBQ0EsWUFBTU0saUJBQWlCO0FBQ25CQyxpQkFBSyxpREFEYztBQUVuQnhGLGtCQUFNO0FBQ0Z5RiwwQkFBVW5CLFFBRFI7QUFFRm9CLDhCQUFjbEIsZ0JBRlo7QUFHRm1CLDBCQUFVUixPQUhSO0FBSUZTLGdDQUFnQlI7QUFKZDtBQUZhLFNBQXZCOztBQVVBO0FBQ0EsWUFBTVMsa0JBQWtCLFNBQWxCQSxlQUFrQixHQUFNO0FBQzFCO0FBQ0F6RixxQkFBU2UsaUJBQVQsQ0FBMkIyRSxNQUEzQixDQUFrQzVGLEVBQUUsVUFBRixFQUFjO0FBQzVDeUQscUJBQUtXLFFBRHVDO0FBRTVDeUIsdUJBQU92QixtQkFBbUIsV0FBbkIsR0FBaUM7QUFGSSxhQUFkLENBQWxDOztBQUtBO0FBQ0F3QixzQ0FBMEIxQixRQUExQixFQUFvQ0UsZ0JBQXBDO0FBQ0gsU0FURDs7QUFXQTtBQUNBMUUsWUFBSTRFLElBQUosQ0FBU3VCLEdBQVQsQ0FBYUMsR0FBYixDQUFpQlgsY0FBakIsRUFBaUNZLElBQWpDLENBQXNDLG9CQUFZO0FBQzlDO0FBQ0E5QyxzQkFBVStDLFdBQVYsQ0FBc0JuQixZQUF0Qjs7QUFFQSxnQkFBSW9CLFNBQVNDLE1BQWIsRUFBcUI7QUFDakI7QUFDQSxvQkFBTUMsaUJBQWlCbkcsU0FBU1UsTUFBVCxDQUFnQkMsV0FBaEIsQ0FBNEJSLElBQTVCLENBQWlDLGdCQUFqQyxDQUF2Qjs7QUFFQTtBQUNBSCx5QkFBU1UsTUFBVCxDQUFnQkMsV0FBaEIsQ0FBNEI0RCxLQUE1QixDQUFrQyxNQUFsQzs7QUFFQTtBQUNBNEIsK0JBQ0tDLEdBREwsQ0FDUyxPQURULEVBRUtDLEVBRkwsQ0FFUSxPQUZSLEVBRWlCWixlQUZqQjtBQUdILGFBWEQsTUFXTztBQUNIQTtBQUNIO0FBQ0osU0FsQkQ7QUFtQkg7O0FBRUQ7Ozs7O0FBS0EsYUFBU2EsV0FBVCxDQUFxQnZELEtBQXJCLEVBQTRCO0FBQ3hCO0FBQ0EsWUFBTWlDLGVBQWVsRixFQUFFaUQsTUFBTUcsTUFBUixFQUNoQkUsT0FEZ0IsQ0FDUmxDLFVBQVVjLGdCQURGLEVBRWhCcEMsSUFGZ0IsQ0FFWCxJQUZXLENBQXJCOztBQUlBO0FBQ0EsWUFBTTBELFFBQVF4RCxFQUFFaUQsTUFBTUcsTUFBUixFQUNURSxPQURTLENBQ0RsQyxVQUFVYyxnQkFEVCxFQUVUN0IsSUFGUyxDQUVKZSxVQUFVZSxnQkFGTixDQUFkOztBQUlBO0FBQ0EsWUFBTWdCLFlBQVluRCxFQUFFaUQsTUFBTUcsTUFBUixFQUNiRSxPQURhLENBQ0xsQyxVQUFVYyxnQkFETCxFQUViN0IsSUFGYSxDQUVSZSxVQUFVTyxNQUFWLENBQWlCQyxRQUZULENBQWxCOztBQUlBO0FBQ0EsWUFBTTZFLGdCQUFnQnRELFVBQVVNLEdBQVYsRUFBdEI7O0FBRUE7QUFDQSxZQUFNaUQscUJBQXFCdkQsVUFBVTlDLElBQVYsQ0FBZSxpQkFBZixFQUFrQ3NHLFFBQWxDLENBQTJDekYsUUFBUUMsUUFBbkQsQ0FBM0I7O0FBRUE7QUFDQSxZQUFNeUYsV0FBV2hILElBQUlpRCxJQUFKLENBQVNnRSxNQUFULENBQWdCYixHQUFoQixDQUFvQixRQUFwQixJQUFnQyx3QkFBaEMsR0FBMkRTLGFBQTVFOztBQUVBO0FBQ0EsWUFBSSxDQUFDdkIsWUFBRCxJQUFpQndCLGtCQUFyQixFQUF5QztBQUNyQzlHLGdCQUFJNEUsSUFBSixDQUFTQyxLQUFULENBQWVDLFdBQWYsQ0FDSTlFLElBQUlpRCxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3Qix1QkFBeEIsRUFBaUQsU0FBakQsQ0FESixFQUVJbkQsSUFBSWlELElBQUosQ0FBU0MsSUFBVCxDQUFjQyxTQUFkLENBQXdCLCtCQUF4QixFQUF5RCxTQUF6RCxDQUZKOztBQUtBO0FBQ0g7O0FBRUQ7QUFDQTdDLGlCQUFTVSxNQUFULENBQWdCRyxRQUFoQixDQUF5QitDLE9BQXpCLENBQWlDLE1BQWpDLEVBQXlDLENBQUNOLEtBQUQsRUFBUW9ELFFBQVIsQ0FBekM7QUFDSDs7QUFFRDs7O0FBR0EsYUFBU0Usa0JBQVQsR0FBOEI7QUFDMUI7QUFDQSxZQUFNQyxpQkFBaUIscUJBQXZCOztBQUVBO0FBQ0EsWUFBTUMseUJBQXlCLGFBQS9COztBQUVBO0FBQ0EsWUFBTUMsYUFBYS9HLFNBQVNjLFNBQVQsQ0FDZDBDLFFBRGMsR0FFZHdELEdBRmMsQ0FFVixTQUZVLENBQW5COztBQUlBO0FBQ0EsWUFBTUMsV0FBV2pILFNBQVNDLE9BQVQsQ0FBaUJDLElBQWpCLENBQXNCdUcsUUFBdEIsQ0FBK0JJLGNBQS9CLENBQWpCOztBQUVBO0FBQ0EsWUFBTUssWUFBWWxILFNBQVNDLE9BQVQsQ0FBaUJDLElBQWpCLENBQXNCTixJQUF0QixDQUEyQixXQUEzQixDQUFsQjtBQUNBLFlBQU11SCxXQUFXbkgsU0FBU0MsT0FBVCxDQUFpQkMsSUFBakIsQ0FBc0JOLElBQXRCLENBQTJCLFVBQTNCLENBQWpCOztBQUVBO0FBQ0EsWUFBTXdILFVBQVV2SCxNQUFNTSxJQUFOLENBQVdlLFVBQVVULFVBQXJCLENBQWhCOztBQUVBO0FBQ0EyRyxnQkFDS0MsSUFETCxHQUVLQyxNQUZMOztBQUlBO0FBQ0F0SCxpQkFBU0MsT0FBVCxDQUFpQkMsSUFBakIsQ0FBc0IrRyxXQUFXLGFBQVgsR0FBMkIsVUFBakQsRUFBNkRKLGNBQTdELEVBQ0tVLElBREwsQ0FDVU4sV0FBV0MsU0FBWCxHQUF1QkMsUUFEakM7O0FBR0E7QUFDQW5ILGlCQUFTQyxPQUFULENBQWlCRyxNQUFqQixDQUF3Qm9ILElBQXhCLENBQTZCLFVBQTdCLEVBQXlDLENBQUNQLFFBQTFDOztBQUVBO0FBQ0FHLGdCQUFRakgsSUFBUixDQUFhZSxVQUFVQyxLQUFWLENBQWdCRSxJQUE3QixFQUFtQzRGLFdBQVcsTUFBWCxHQUFvQixNQUF2RDs7QUFFQTtBQUNBRixtQkFBV0UsV0FBVyxRQUFYLEdBQXNCLFNBQWpDOztBQUVBO0FBQ0FHLGdCQUNLakgsSUFETCxDQUNVZSxVQUFVQyxLQUFWLENBQWdCRyxTQUQxQixFQUVLbUcsR0FGTCxDQUVTdkcsVUFBVUMsS0FBVixDQUFnQkMsTUFGekIsRUFFaUM2RixXQUFXLE1BQVgsR0FBb0IsTUFGckQ7O0FBSUE7QUFDQSxZQUFJLENBQUNBLFFBQUwsRUFBZTtBQUNYL0UsOEJBQWtCa0YsUUFBUTNELE1BQVIsQ0FBZSxVQUFDaUUsS0FBRCxFQUFRQyxPQUFSO0FBQUEsdUJBQW9CN0gsRUFBRTZILE9BQUYsRUFDaER4SCxJQURnRCxDQUMzQzJHLHNCQUQyQyxFQUVoRDdCLEVBRmdELENBRTdDLFVBRjZDLENBQXBCO0FBQUEsYUFBZixDQUFsQjtBQUdIOztBQUVEO0FBQ0EvQyx3QkFBZ0IwRixJQUFoQixDQUFxQixVQUFDRixLQUFELEVBQVFDLE9BQVI7QUFBQSxtQkFBb0I3SCxFQUFFNkgsT0FBRixFQUNwQ3hILElBRG9DLENBQy9CZSxVQUFVQyxLQUFWLENBQWdCRyxTQURlLEVBRXBDc0MsT0FGb0MsQ0FFNUIsT0FGNEIsQ0FBcEI7QUFBQSxTQUFyQjtBQUdIOztBQUVEOzs7QUFHQSxhQUFTaUUsWUFBVCxHQUF3QjtBQUNwQjtBQUNBLFlBQU1DLFdBQVdqSSxNQUNaTSxJQURZLENBQ1BlLFVBQVVZLE9BREgsRUFFWjJCLE1BRlksQ0FFTCxVQUZLLENBQWpCOztBQUlBO0FBQ0FxRSxpQkFBU0MsUUFBVCxDQUFrQixrQkFBbEI7QUFDSDs7QUFFRDs7O0FBR0EsYUFBU0MsV0FBVCxHQUF1QjtBQUNuQjtBQUNBN0Y7QUFDSDs7QUFFRDs7Ozs7QUFLQSxhQUFTOEYsa0JBQVQsQ0FBNEJsRixLQUE1QixFQUFtQztBQUMvQjtBQUNBLFlBQU1tRixjQUFjcEksRUFBRWlELE1BQU1HLE1BQVIsRUFBZ0JFLE9BQWhCLENBQXdCbEMsVUFBVVQsVUFBbEMsQ0FBcEI7O0FBRUE7QUFDQSxZQUFNMEYsaUJBQWlCbkcsU0FBU1UsTUFBVCxDQUFnQkUsV0FBaEIsQ0FBNEJULElBQTVCLENBQWlDLGdCQUFqQyxDQUF2Qjs7QUFFQTtBQUNBSCxpQkFBU1UsTUFBVCxDQUFnQkUsV0FBaEIsQ0FBNEIyRCxLQUE1QixDQUFrQyxNQUFsQzs7QUFFQTtBQUNBNEIsdUJBQ0tDLEdBREwsQ0FDUyxPQURULEVBRUtDLEVBRkwsQ0FFUSxPQUZSLEVBRWlCO0FBQUEsbUJBQU04QixpQ0FBaUNuSSxTQUFTVSxNQUFULENBQWdCRSxXQUFqRCxFQUE4RHNILFdBQTlELENBQU47QUFBQSxTQUZqQjtBQUdIOztBQUVEOzs7QUFHQSxhQUFTRSxvQkFBVCxHQUFnQztBQUM1QjtBQUNBLFlBQU1GLGNBQWNwSSxFQUFFRSxTQUFTUSxTQUFULENBQW1CQyxVQUFuQixDQUE4QjRILEtBQTlCLEdBQXNDQyxJQUF0QyxFQUFGLENBQXBCOztBQUVBO0FBQ0EsWUFBTVIsV0FBV2pJLE1BQ1pNLElBRFksQ0FDUGUsVUFBVVksT0FESCxFQUVaMkIsTUFGWSxDQUVMLFVBRkssQ0FBakI7O0FBSUE7QUFDQSxZQUFNOEUsZUFBZVQsU0FBUzNILElBQVQsQ0FBY2UsVUFBVVQsVUFBeEIsQ0FBckI7O0FBRUE7QUFDQSxZQUFNK0gsYUFBYUQsYUFBYXhFLE1BQWIsR0FBc0IsQ0FBekM7O0FBRUE7QUFDQSxZQUFNMEUsV0FBYy9JLElBQUlpRCxJQUFKLENBQVNDLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixXQUF4QixFQUFxQyxTQUFyQyxDQUFkLFNBQWlFMkYsVUFBdkU7O0FBRUE7QUFDQU4sb0JBQ0svSCxJQURMLENBQ1VlLFVBQVVhLGVBRHBCLEVBRUt3RixJQUZMLENBRVVrQixRQUZWOztBQUlBO0FBQ0FQLG9CQUNLL0gsSUFETCxDQUNVZSxVQUFVTyxNQUFWLENBQWlCRyxLQUQzQixFQUVLMkIsR0FGTCxDQUVTa0YsUUFGVDs7QUFJQTtBQUNBLFlBQUlGLGFBQWF4RSxNQUFqQixFQUF5QjtBQUNyQjtBQUNBLGdCQUFNMkUsYUFBYUgsYUFDZEksS0FEYyxHQUVkeEksSUFGYyxDQUVUZSxVQUFVTyxNQUFWLENBQWlCQyxRQUZSLENBQW5COztBQUlBO0FBQ0EsZ0JBQU1rSCxvQkFBb0JGLFdBQ3JCakYsTUFEcUIsQ0FDZHZDLFVBQVVPLE1BQVYsQ0FBaUJFLHNCQURILEVBRXJCNkIsUUFGcUIsR0FHckI2RSxLQUhxQixFQUExQjs7QUFLQTtBQUNBLGdCQUFNUSxnQkFBZ0JILFdBQ2pCMUIsR0FEaUIsQ0FDYjlGLFVBQVVPLE1BQVYsQ0FBaUJFLHNCQURKLEVBRWpCZ0gsS0FGaUIsR0FHakJuRixRQUhpQixHQUlqQjZFLEtBSmlCLEVBQXRCOztBQU1BO0FBQ0FILHdCQUNLL0gsSUFETCxDQUNVZSxVQUFVTyxNQUFWLENBQWlCRSxzQkFEM0IsRUFFS21ILEtBRkwsR0FHS3BELE1BSEwsQ0FHWWtELGlCQUhaLEVBSUtyRixHQUpMLENBSVMsRUFKVDs7QUFNQTtBQUNBMkUsd0JBQ0svSCxJQURMLENBQ1VlLFVBQVVPLE1BQVYsQ0FBaUJDLFFBRDNCLEVBRUtzRixHQUZMLENBRVM5RixVQUFVTyxNQUFWLENBQWlCRSxzQkFGMUIsRUFHS21ILEtBSEwsR0FJS3BELE1BSkwsQ0FJWW1ELGFBSlosRUFLS3RGLEdBTEwsQ0FLUyxFQUxUO0FBTUg7O0FBRUQ7QUFDQTJFLG9CQUNLYixJQURMLEdBRUswQixTQUZMLENBRWVqQixRQUZmLEVBR0tSLE1BSEw7O0FBS0E7QUFDQS9ILFdBQUd5SixPQUFILENBQVdDLElBQVgsQ0FBZ0JmLFdBQWhCO0FBQ0EzSSxXQUFHMkosVUFBSCxDQUFjRCxJQUFkLENBQW1CZixXQUFuQjs7QUFFQTtBQUNBQSxvQkFDSy9ILElBREwsQ0FDVSxRQURWLEVBRUt5RCxPQUZMLENBRWEsUUFGYjs7QUFJQTtBQUNBekI7O0FBRUE7QUFDQWdIO0FBQ0g7O0FBRUQ7Ozs7OztBQU1BLGFBQVNoQixnQ0FBVCxDQUEwQ2lCLE1BQTFDLEVBQWtEbEIsV0FBbEQsRUFBK0Q7QUFDM0Q7QUFDQWtCLGVBQU83RSxLQUFQLENBQWEsTUFBYjs7QUFFQTtBQUNBMkQsb0JBQVltQixPQUFaLENBQW9CLEdBQXBCLEVBQXlCLFlBQU07QUFDM0I7QUFDQW5CLHdCQUFZeEUsTUFBWjs7QUFFQTtBQUNBeUY7QUFDSCxTQU5EOztBQVFBO0FBQ0FoSDtBQUNIOztBQUVEOzs7OztBQUtBLGFBQVNtSCxhQUFULENBQXVCdkcsS0FBdkIsRUFBOEI7QUFDMUI7QUFDQSxZQUFNd0csU0FBU3pKLEVBQUVpRCxNQUFNRyxNQUFSLENBQWY7O0FBRUE7QUFDQSxZQUFNc0csU0FBU0QsT0FDVm5HLE9BRFUsQ0FDRmxDLFVBQVVULFVBRFIsRUFFVk4sSUFGVSxDQUVMZSxVQUFVYSxlQUZMLENBQWY7O0FBSUE7QUFDQXlILGVBQU9qQyxJQUFQLENBQVlnQyxPQUFPaEcsR0FBUCxFQUFaO0FBQ0g7O0FBRUQ7Ozs7O0FBS0EsYUFBU2tHLHNCQUFULENBQWdDMUcsS0FBaEMsRUFBdUM7QUFDbkM7QUFDQSxZQUFNMkcsT0FBTzVKLEVBQUVpRCxNQUFNRyxNQUFSLENBQWI7O0FBRUE7QUFDQSxZQUFNRyxnQkFBZ0JxRyxLQUFLdkosSUFBTCxDQUFVZSxVQUFVQyxLQUFWLENBQWdCTixRQUExQixDQUF0Qjs7QUFFQTtBQUNBLFlBQU15QyxRQUFRb0csS0FBS3ZKLElBQUwsQ0FBVWUsVUFBVWUsZ0JBQXBCLENBQWQ7O0FBRUE7QUFDQSxZQUFJLENBQUNvQixjQUFjVSxNQUFmLElBQXlCLENBQUNULE1BQU1TLE1BQXBDLEVBQTRDO0FBQ3hDO0FBQ0g7O0FBRUQsWUFBSVQsTUFBTUUsUUFBTixHQUFpQk8sTUFBckIsRUFBNkI7QUFDekJWLDBCQUFjMkMsV0FBZCxDQUEwQixrQkFBMUIsRUFBOENkLFFBQTlDLENBQXVELHlCQUF2RDtBQUNILFNBRkQsTUFFTztBQUNIN0IsMEJBQWMyQyxXQUFkLENBQTBCLHlCQUExQixFQUFxRGQsUUFBckQsQ0FBOEQsa0JBQTlEO0FBQ0g7QUFDSjs7QUFFRDs7O0FBR0EsYUFBU3lFLGFBQVQsR0FBeUI7QUFDckI5SixjQUNLdUQsT0FETCxDQUNhLE1BRGIsRUFFS1EsT0FGTCxDQUVhLFFBRmI7QUFHSDs7QUFFRDs7O0FBR0EsYUFBU2dHLGdCQUFULEdBQTRCO0FBQ3hCL0osY0FDS3VELE9BREwsQ0FDYSxNQURiLEVBRUtRLE9BRkwsQ0FFYSxRQUZiLEVBRXVCLEVBQUNpRyxTQUFTLElBQVYsRUFGdkI7QUFHSDs7QUFFRDs7Ozs7O0FBTUEsYUFBU2xGLG9CQUFULENBQThCVCxRQUE5QixFQUFxRTtBQUFBLFlBQTdCNEYsbUJBQTZCLHVFQUFQLEtBQU87O0FBQ2pFO0FBQ0EsWUFBTXBCLGFBQWE3SSxNQUNkTSxJQURjLENBQ1RlLFVBQVVPLE1BQVYsQ0FBaUJDLFFBRFIsRUFFZG9JLHNCQUFzQixRQUF0QixHQUFpQyxLQUZuQixFQUUwQjVJLFVBQVVPLE1BQVYsQ0FBaUJFLHNCQUYzQyxDQUFuQjs7QUFJQTtBQUNBLFlBQU04QyxVQUFVM0UsRUFBRSxVQUFGLEVBQWMsRUFBQ2lLLE9BQU83RixRQUFSLEVBQWtCcUQsTUFBTXJELFFBQXhCLEVBQWtDeUIsT0FBTzNFLFFBQVFDLFFBQWpELEVBQWQsQ0FBaEI7O0FBRUE7QUFDQXlILG1CQUFXaEQsTUFBWCxDQUFrQmpCLE9BQWxCO0FBQ0g7O0FBRUQ7Ozs7OztBQU1BLGFBQVNtQix5QkFBVCxDQUFtQzFCLFFBQW5DLEVBQTBFO0FBQUEsWUFBN0I0RixtQkFBNkIsdUVBQVAsS0FBTzs7QUFDdEU7QUFDQSxZQUFNcEIsYUFBYTdJLE1BQ2RNLElBRGMsQ0FDVGUsVUFBVU8sTUFBVixDQUFpQkMsUUFEUixFQUVkb0ksc0JBQXNCLFFBQXRCLEdBQWlDLEtBRm5CLEVBRTBCNUksVUFBVU8sTUFBVixDQUFpQkUsc0JBRjNDLENBQW5COztBQUlBO0FBQ0ErRyxtQkFBV2QsSUFBWCxDQUFnQixVQUFDRixLQUFELEVBQVFDLE9BQVIsRUFBb0I7QUFDaEM7QUFDQSxnQkFBTTFFLFlBQVluRCxFQUFFNkgsT0FBRixDQUFsQjs7QUFFQTtBQUNBMUUsc0JBQVU5QyxJQUFWLGNBQTBCK0QsUUFBMUIsU0FBd0NSLE1BQXhDOztBQUVBO0FBQ0EsZ0JBQUlULFVBQVVPLFFBQVYsR0FBcUJPLE1BQXJCLElBQStCLENBQW5DLEVBQXNDO0FBQ2xDZCwwQkFBVU0sR0FBVixDQUFjLEVBQWQ7QUFDSDs7QUFFRDtBQUNBTixzQkFBVVcsT0FBVixDQUFrQixRQUFsQjtBQUNILFNBZEQ7QUFlSDs7QUFFRDs7O0FBR0EsYUFBU3VGLG1CQUFULEdBQStCO0FBQzNCO0FBQ0EsWUFBSWEsd0JBQXdCLElBQTVCOztBQUVBO0FBQ0EsWUFBTTVDLFVBQVV2SCxNQUFNTSxJQUFOLENBQVdlLFVBQVVULFVBQXJCLENBQWhCOztBQUVBO0FBQ0EsWUFBSSxDQUFDMkcsUUFBUXJELE1BQWIsRUFBcUI7QUFDakJpRyxvQ0FBd0IsS0FBeEI7QUFDSDs7QUFFRDtBQUNBaEssaUJBQVNDLE9BQVQsQ0FBaUJJLE1BQWpCLENBQ0ttRCxRQURMLEdBRUt3RCxHQUZMLENBRVMsSUFGVCxFQUdLUSxJQUhMLENBR1UsVUFIVixFQUdzQixDQUFDd0MscUJBSHZCO0FBSUg7O0FBRUQ7QUFDQTtBQUNBOztBQUVBdkssV0FBT3dKLElBQVAsR0FBYyxnQkFBUTtBQUNsQjtBQUNBakosaUJBQVNDLE9BQVQsQ0FBaUJDLElBQWpCLENBQXNCbUcsRUFBdEIsQ0FBeUIsT0FBekIsRUFBa0NPLGtCQUFsQzs7QUFFQTtBQUNBL0csY0FDS3dHLEVBREwsQ0FDUSxXQURSLEVBQ3FCd0IsWUFEckIsRUFFS3hCLEVBRkwsQ0FFUSxVQUZSLEVBRW9CMkIsV0FGcEIsRUFHSzNCLEVBSEwsQ0FHUSxPQUhSLEVBR2lCbkYsVUFBVUMsS0FBVixDQUFnQkMsTUFIakMsRUFHeUM2RyxrQkFIekMsRUFJSzVCLEVBSkwsQ0FJUSxPQUpSLEVBSWlCbkYsVUFBVU8sTUFBVixDQUFpQkcsS0FKbEMsRUFJeUMwSCxhQUp6QyxFQUtLakQsRUFMTCxDQUtRLFFBTFIsRUFLa0JuRixVQUFVTyxNQUFWLENBQWlCSSxJQUxuQyxFQUt5Q2dDLFdBTHpDLEVBTUt3QyxFQU5MLENBTVEsT0FOUixFQU1pQm5GLFVBQVVDLEtBQVYsQ0FBZ0JLLE1BTmpDLEVBTXlDbUMsa0JBTnpDLEVBT0swQyxFQVBMLENBT1EsT0FQUixFQU9pQm5GLFVBQVVDLEtBQVYsQ0FBZ0JJLFdBUGpDLEVBTzhDcUQsY0FQOUMsRUFRS3lCLEVBUkwsQ0FRUSxPQVJSLEVBUWlCbkYsVUFBVUMsS0FBVixDQUFnQk4sUUFSakMsRUFRMkN5RixXQVIzQyxFQVNLRCxFQVRMLENBU1EsUUFUUixFQVNrQm5GLFVBQVVPLE1BQVYsQ0FBaUJDLFFBVG5DLEVBUzZDb0IsY0FUN0MsRUFVS3VELEVBVkwsQ0FVUSxZQVZSLEVBVXNCbkYsVUFBVWMsZ0JBVmhDLEVBVWtEeUgsc0JBVmxEOztBQVlBO0FBQ0F6SixpQkFBU0MsT0FBVCxDQUFpQkssVUFBakIsQ0FBNEIrRixFQUE1QixDQUErQixPQUEvQixFQUF3Q3NELGFBQXhDO0FBQ0EzSixpQkFBU0MsT0FBVCxDQUFpQk0sYUFBakIsQ0FBK0I4RixFQUEvQixDQUFrQyxPQUFsQyxFQUEyQ3VELGdCQUEzQzs7QUFFQTtBQUNBNUosaUJBQVNDLE9BQVQsQ0FBaUJHLE1BQWpCLENBQXdCaUcsRUFBeEIsQ0FBMkIsT0FBM0IsRUFBb0MrQixvQkFBcEM7O0FBRUE7QUFDQXBJLGlCQUFTYyxTQUFULENBQ0swQyxRQURMLEdBRUttRixLQUZMLEdBR0t6RCxRQUhMLENBR2MsUUFIZDs7QUFLQTtBQUNBckYsY0FDS00sSUFETCxDQUNVZSxVQUFVWSxPQURwQixFQUVLNkcsS0FGTCxHQUdLekQsUUFITCxDQUdjLFdBSGQ7O0FBS0E7QUFDQXJGLGNBQ0tNLElBREwsQ0FDVWUsVUFBVU8sTUFBVixDQUFpQkMsUUFEM0IsRUFFS2tDLE9BRkwsQ0FFYSxRQUZiLEVBRXVCLENBQUMsS0FBRCxDQUZ2Qjs7QUFJQTtBQUNBcUc7QUFDSCxLQTNDRDs7QUE2Q0EsV0FBT3hLLE1BQVA7QUFDSCxDQTl5QkwiLCJmaWxlIjoic2xpZGVycy9kZXRhaWxzL3NsaWRlc19jb250YWluZXIuanMiLCJzb3VyY2VzQ29udGVudCI6WyIvKiAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuIHNsaWRlc19jb250YWluZXIuanMgMjAxNi0xMi0yOVxuIEdhbWJpbyBHbWJIXG4gaHR0cDovL3d3dy5nYW1iaW8uZGVcbiBDb3B5cmlnaHQgKGMpIDIwMTYgR2FtYmlvIEdtYkhcbiBSZWxlYXNlZCB1bmRlciB0aGUgR05VIEdlbmVyYWwgUHVibGljIExpY2Vuc2UgKFZlcnNpb24gMilcbiBbaHR0cDovL3d3dy5nbnUub3JnL2xpY2Vuc2VzL2dwbC0yLjAuaHRtbF1cbiAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICovXG5cbi8qKlxuICogQ29udHJvbGxlciBNb2R1bGUgRm9yIFNsaWRlcyBDb250YWluZXIgKFRhYnMpXG4gKlxuICogSGFuZGxlcyB0aGUgc2xpZGVycyBjb250YWluZXIgZnVuY3Rpb25hbGl0eSBpbiB0aGUgc2xpZGVycyBkZXRhaWxzIHBhZ2UuXG4gKi9cbmd4LmNvbnRyb2xsZXJzLm1vZHVsZShcbiAgICAnc2xpZGVzX2NvbnRhaW5lcicsXG5cbiAgICBbXG4gICAgICAgIGAke2pzZS5zb3VyY2V9L3ZlbmRvci9qcXVlcnktZGVwYXJhbS9qcXVlcnktZGVwYXJhbS5taW4uanNgLFxuICAgICAgICBgJHtqc2Uuc291cmNlfS92ZW5kb3IvanF1ZXJ5LXVpLWRpc3QvanF1ZXJ5LXVpLm1pbi5jc3NgLFxuICAgICAgICBgJHtqc2Uuc291cmNlfS92ZW5kb3IvanF1ZXJ5LXVpLWRpc3QvanF1ZXJ5LXVpLm1pbi5qc2AsXG4gICAgICAgICd4aHInLFxuICAgICAgICAnbW9kYWwnXG4gICAgXSxcblxuICAgIGZ1bmN0aW9uIChkYXRhKSB7XG5cbiAgICAgICAgJ3VzZSBzdHJpY3QnO1xuXG4gICAgICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4gICAgICAgIC8vIFZBUklBQkxFU1xuICAgICAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiBNb2R1bGUgU2VsZWN0b3JcbiAgICAgICAgICpcbiAgICAgICAgICogQHR5cGUge2pRdWVyeX1cbiAgICAgICAgICovXG4gICAgICAgIGNvbnN0ICR0aGlzID0gJCh0aGlzKTtcbiAgICAgICAgY29uc3QgJGZvb3RlciA9ICQoJyNtYWluLWZvb3RlcicpO1xuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiBFbGVtZW50c1xuICAgICAgICAgKlxuICAgICAgICAgKiBAdHlwZSB7T2JqZWN0fVxuICAgICAgICAgKi9cbiAgICAgICAgY29uc3QgZWxlbWVudHMgPSB7XG4gICAgICAgICAgICAvLyBCdXR0b25zLlxuICAgICAgICAgICAgYnV0dG9uczoge1xuICAgICAgICAgICAgICAgIC8vIFNvcnQgbW9kZSBidXR0b24uXG4gICAgICAgICAgICAgICAgc29ydDogJHRoaXMuZmluZCgnLnNvcnQtYnV0dG9uJyksXG5cbiAgICAgICAgICAgICAgICAvLyBDcmVhdGUgYnV0dG9uLlxuICAgICAgICAgICAgICAgIGNyZWF0ZTogJHRoaXMuZmluZCgnLmJ0bi1jcmVhdGUnKSxcblxuICAgICAgICAgICAgICAgIC8vIFN1Ym1pdCBidXR0b24gZ3JvdXAuXG4gICAgICAgICAgICAgICAgc3VibWl0OiAkZm9vdGVyLmZpbmQoJy5zdWJtaXQtYnV0dG9uLWdyb3VwJyksXG5cbiAgICAgICAgICAgICAgICAvLyBTdWJtaXQgYnV0dG9uIGZvciBzYXZlIHNsaWRlclxuICAgICAgICAgICAgICAgIHN1Ym1pdFNhdmU6ICRmb290ZXIuZmluZCgnLnNhdmUnKSxcblxuICAgICAgICAgICAgICAgIC8vIFN1Ym1pdCBidXR0b24gZm9yIHNhdmUgYW5kIHJlZnJlc2ggc2xpZGVyXG4gICAgICAgICAgICAgICAgc3VibWl0UmVmcmVzaDogJGZvb3Rlci5maW5kKCcucmVmcmVzaCcpLFxuICAgICAgICAgICAgfSxcblxuICAgICAgICAgICAgLy8gVGVtcGxhdGUuXG4gICAgICAgICAgICB0ZW1wbGF0ZXM6IHtcbiAgICAgICAgICAgICAgICAvLyBTbGlkZSBwYW5lbCBzZXQgdGVtcGxhdGUuXG4gICAgICAgICAgICAgICAgc2xpZGVQYW5lbDogJHRoaXMuZmluZCgnI3NsaWRlLXBhbmVsLXRlbXBsYXRlJylcbiAgICAgICAgICAgIH0sXG5cbiAgICAgICAgICAgIC8vIE1vZGFscy5cbiAgICAgICAgICAgIG1vZGFsczoge1xuICAgICAgICAgICAgICAgIC8vIERlbGV0ZSBpbWFnZSBtb2RhbC5cbiAgICAgICAgICAgICAgICBkZWxldGVJbWFnZTogJCgnLmRlbGV0ZS1pbWFnZS5tb2RhbCcpLFxuXG4gICAgICAgICAgICAgICAgLy8gRGVsZXRlIHNsaWRlIG1vZGFsLlxuICAgICAgICAgICAgICAgIGRlbGV0ZVNsaWRlOiAkKCcuZGVsZXRlLXNsaWRlLm1vZGFsJyksXG5cbiAgICAgICAgICAgICAgICAvLyBFZGl0IGltYWdlIG1hcCBtb2RhbC5cbiAgICAgICAgICAgICAgICBpbWFnZU1hcDogJCgnLmltYWdlLW1hcC5tb2RhbCcpLFxuICAgICAgICAgICAgfSxcblxuICAgICAgICAgICAgLy8gVGFicy5cbiAgICAgICAgICAgIHRhYkhlYWRlcjogJHRoaXMuZmluZCgnLm5hdi10YWJzJyksXG5cbiAgICAgICAgICAgIC8vIFNlbGVjdCBib3ggd2hpY2ggaG9sZHMgYWxsIGltYWdlcyB0aGF0IHdpbGwgYmUgZGVsZXRlZC5cbiAgICAgICAgICAgIGRlbGV0ZUltYWdlU2VsZWN0OiAkKCcjZGVsZXRlX2ltYWdlcycpXG4gICAgICAgIH07XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIENTUyBjbGFzcyBuYW1lcy5cbiAgICAgICAgICpcbiAgICAgICAgICogQHR5cGUge09iamVjdH1cbiAgICAgICAgICovXG4gICAgICAgIGNvbnN0IGNsYXNzZXMgPSB7XG4gICAgICAgICAgICAvLyBOZXcgaW1hZ2UuXG4gICAgICAgICAgICBuZXdJbWFnZTogJ25ldydcbiAgICAgICAgfTtcblxuICAgICAgICAvKipcbiAgICAgICAgICogU2VsZWN0b3IgU3RyaW5nc1xuICAgICAgICAgKlxuICAgICAgICAgKiBAdHlwZSB7T2JqZWN0fVxuICAgICAgICAgKi9cbiAgICAgICAgY29uc3Qgc2VsZWN0b3JzID0ge1xuICAgICAgICAgICAgLy8gSWNvbiBzZWxlY3RvciBzdHJpbmdzLlxuICAgICAgICAgICAgaWNvbnM6IHtcbiAgICAgICAgICAgICAgICAvLyBEZWxldGUgYnV0dG9uIG9uIHRoZSBwYW5lbCBoZWFkZXIuXG4gICAgICAgICAgICAgICAgZGVsZXRlOiAnLmljb24uZGVsZXRlJyxcblxuICAgICAgICAgICAgICAgIC8vIERyYWcgYnV0dG9uIG9uIHRoZSBwYW5lbCBoZWFkZXIuXG4gICAgICAgICAgICAgICAgZHJhZzogJy5kcmFnLWhhbmRsZScsXG5cbiAgICAgICAgICAgICAgICAvLyBDb2xsYXBzZXIgYnV0dG9uIG9uIHRoZSBwYW5lbCBoZWFkZXIuXG4gICAgICAgICAgICAgICAgY29sbGFwc2VyOiAnLmNvbGxhcHNlcicsXG5cbiAgICAgICAgICAgICAgICAvLyBJbWFnZSBkZWxldGUgYnV0dG9uLlxuICAgICAgICAgICAgICAgIGltYWdlRGVsZXRlOiAnLmFjdGlvbi1pY29uLmRlbGV0ZScsXG5cbiAgICAgICAgICAgICAgICAvLyBJbWFnZSBtYXAgZWRpdCBidXR0b24uXG4gICAgICAgICAgICAgICAgaW1hZ2VNYXA6ICcuYWN0aW9uLWljb24uaW1hZ2UtbWFwJyxcblxuICAgICAgICAgICAgICAgIC8vIFVwbG9hZCBpbWFnZSBidXR0b24uXG4gICAgICAgICAgICAgICAgdXBsb2FkOiAnLmFjdGlvbi1pY29uLnVwbG9hZCdcbiAgICAgICAgICAgIH0sXG5cbiAgICAgICAgICAgIC8vIElucHV0cyBzZWxlY3RvciBzdHJpbmdzLlxuICAgICAgICAgICAgaW5wdXRzOiB7XG4gICAgICAgICAgICAgICAgLy8gR2VuZXJhbCBpbWFnZSBzZWxlY3QgZHJvcGRvd25zLlxuICAgICAgICAgICAgICAgIGRyb3Bkb3duOiAnLmRyb3Bkb3duLWlucHV0JyxcblxuICAgICAgICAgICAgICAgIC8vIFRodW1ibmFpbCBkcm9wZG93bi5cbiAgICAgICAgICAgICAgICB0aHVtYm5haWxJbWFnZURyb3Bkb3duOiAnW25hbWU9XCJ0aHVtYm5haWxcIl0nLFxuXG4gICAgICAgICAgICAgICAgLy8gVGl0bGUuXG4gICAgICAgICAgICAgICAgdGl0bGU6ICdpbnB1dFtuYW1lPVwidGl0bGVcIl0nLFxuXG4gICAgICAgICAgICAgICAgLy8gRmlsZS5cbiAgICAgICAgICAgICAgICBmaWxlOiAnLmZpbGUtaW5wdXQnXG4gICAgICAgICAgICB9LFxuXG4gICAgICAgICAgICAvLyBTbGlkZSBwYW5lbC5cbiAgICAgICAgICAgIHNsaWRlUGFuZWw6ICcucGFuZWwnLFxuXG4gICAgICAgICAgICAvLyBUYWIgYm9keS5cbiAgICAgICAgICAgIHRhYkJvZHk6ICcudGFiLXBhbmUnLFxuXG4gICAgICAgICAgICAvLyBTbGlkZSBwYW5lbCB0aXRsZS5cbiAgICAgICAgICAgIHNsaWRlUGFuZWxUaXRsZTogJy5zbGlkZS10aXRsZScsXG5cbiAgICAgICAgICAgIC8vIFNldHRpbmcgcm93IChmb3JtIGdyb3VwKS5cbiAgICAgICAgICAgIGNvbmZpZ3VyYXRpb25Sb3c6ICcucm93LmZvcm0tZ3JvdXAnLFxuXG4gICAgICAgICAgICAvLyBEYXRhIGxpc3QgY29udGFpbmVyIGZvciBpbWFnZSBtYXAuXG4gICAgICAgICAgICBpbWFnZU1hcERhdGFMaXN0OiAnLmltYWdlLW1hcC1kYXRhLWxpc3QnLFxuICAgICAgICB9O1xuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiBDYWNoZSBsaXN0IG9mIG9wZW4gc2xpZGUgcGFuZWxzLlxuICAgICAgICAgKlxuICAgICAgICAgKiBAdHlwZSB7alF1ZXJ5W119XG4gICAgICAgICAqL1xuICAgICAgICBsZXQgb3BlblNsaWRlUGFuZWxzID0gW107XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIE1vZHVsZSBJbnN0YW5jZVxuICAgICAgICAgKlxuICAgICAgICAgKiBAdHlwZSB7T2JqZWN0fVxuICAgICAgICAgKi9cbiAgICAgICAgY29uc3QgbW9kdWxlID0ge307XG5cbiAgICAgICAgLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiAgICAgICAgLy8gRlVOQ1RJT05TXG4gICAgICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIFJlZ2lzdGVycyBhIGNoYW5nZSwgc28gdGhhdCB0aGUgdXNlciBnZXRzIGEgY29uZmlybWF0aW9uIGRpYWxvZyB3aGlsZSBsZWF2aW5nIHRoZSBwYWdlLlxuICAgICAgICAgKi9cbiAgICAgICAgZnVuY3Rpb24gX3JlZ2lzdGVyQ2hhbmdlKCkge1xuICAgICAgICAgICAgLy8gT2JqZWN0IG9mIEdFVCBwYXJhbWV0ZXJzLlxuICAgICAgICAgICAgY29uc3QgZ2V0UGFyYW1ldGVycyA9ICQuZGVwYXJhbSh3aW5kb3cubG9jYXRpb24uc2VhcmNoLnNsaWNlKDEpKTtcblxuICAgICAgICAgICAgLy8gT25seSByZWdpc3RlciBpbiBzbGlkZXIgZWRpdCBtb2RlLlxuICAgICAgICAgICAgaWYgKCdpZCcgaW4gZ2V0UGFyYW1ldGVycykge1xuICAgICAgICAgICAgICAgIHdpbmRvdy5vbmJlZm9yZXVubG9hZCA9ICgpID0+IGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdFWElUX0NPTkZJUk1BVElPTl9URVhUJywgJ3NsaWRlcnMnKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiBIYW5kbGVzIHRoZSBpbWFnZSBkcm9wZG93biBjaGFuZ2UgZXZlbnQuXG4gICAgICAgICAqXG4gICAgICAgICAqIEBwYXJhbSB7alF1ZXJ5LkV2ZW50fSBldmVudCBUcmlnZ2VyZWQgZXZlbnQuXG4gICAgICAgICAqIEBwYXJhbSB7Qm9vbGVhbn0gW3JlbW92ZUFsbERhdGFMaXN0SXRlbXMgPSBmYWxzZV0gUmVtb3ZlIGFsbCBkYXRhIGxpc3QgY29udGFpbmVyIGxpc3QgaXRlbXM/XG4gICAgICAgICAqL1xuICAgICAgICBmdW5jdGlvbiBfb25JbWFnZUNoYW5nZShldmVudCwgcmVtb3ZlQWxsRGF0YUxpc3RJdGVtcyA9IHRydWUpIHtcbiAgICAgICAgICAgIC8vIEltYWdlIGRyb3Bkb3duIGVsZW1lbnQuXG4gICAgICAgICAgICBjb25zdCAkZHJvcGRvd24gPSAkKGV2ZW50LnRhcmdldCk7XG5cbiAgICAgICAgICAgIC8vIFJlbW92ZSBpY29uIGVsZW1lbnQuXG4gICAgICAgICAgICBjb25zdCAkcmVtb3ZlSWNvbiA9ICRkcm9wZG93blxuICAgICAgICAgICAgICAgIC5wYXJlbnRzKHNlbGVjdG9ycy5jb25maWd1cmF0aW9uUm93KVxuICAgICAgICAgICAgICAgIC5maW5kKHNlbGVjdG9ycy5pY29ucy5pbWFnZURlbGV0ZSk7XG5cbiAgICAgICAgICAgIC8vIEltYWdlIG1hcCBpY29uIGVsZW1lbnQuXG4gICAgICAgICAgICBjb25zdCAkaW1hZ2VNYXBJY29uID0gJGRyb3Bkb3duXG4gICAgICAgICAgICAgICAgLnBhcmVudHMoc2VsZWN0b3JzLmNvbmZpZ3VyYXRpb25Sb3cpXG4gICAgICAgICAgICAgICAgLmZpbmQoc2VsZWN0b3JzLmljb25zLmltYWdlTWFwKTtcblxuICAgICAgICAgICAgLy8gSW1hZ2UgbWFwIGRhdGEgY29udGFpbmVyIGxpc3QgZWxlbWVudC5cbiAgICAgICAgICAgIGNvbnN0ICRsaXN0ID0gJGRyb3Bkb3duXG4gICAgICAgICAgICAgICAgLnBhcmVudHMoc2VsZWN0b3JzLmNvbmZpZ3VyYXRpb25Sb3cpXG4gICAgICAgICAgICAgICAgLmZpbmQoc2VsZWN0b3JzLmltYWdlTWFwRGF0YUxpc3QpO1xuXG4gICAgICAgICAgICAvLyBSZW1vdmUgdGhlIHJlbW92ZSBpY29uIGlmICdkbyBub3QgdXNlJyBpcyBzZWxlY3RlZC5cbiAgICAgICAgICAgICRyZW1vdmVJY29uWyRkcm9wZG93bi52YWwoKSA/ICdzaG93JyA6ICdoaWRlJ10oKTtcblxuICAgICAgICAgICAgLy8gUmVtb3ZlIHRoZSBpbWFnZSBtYXAgaWNvbiBpZiAnZG8gbm90IHVzZScgaXMgc2VsZWN0ZWQuXG4gICAgICAgICAgICAkaW1hZ2VNYXBJY29uWyRkcm9wZG93bi52YWwoKSA/ICdzaG93JyA6ICdoaWRlJ10oKTtcblxuICAgICAgICAgICAgLy8gRW1wdHkgaW1hZ2UgbWFwIGRhdGEgY29udGFpbmVyIGxpc3QuXG4gICAgICAgICAgICAkbGlzdFxuICAgICAgICAgICAgICAgIC5jaGlsZHJlbigpXG4gICAgICAgICAgICAgICAgLmZpbHRlcihyZW1vdmVBbGxEYXRhTGlzdEl0ZW1zID8gJyonIDogJy5uZXcnKVxuICAgICAgICAgICAgICAgIC5yZW1vdmUoKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiBUcmlnZ2VycyB0aGUgZmlsZSBzZWxlY3QgKGNsaWNrKSBldmVudCBvZiB0aGUgaW52aXNpYmxlIGZpbGUgaW5wdXQgZmllbGQuXG4gICAgICAgICAqXG4gICAgICAgICAqIEBwYXJhbSB7alF1ZXJ5LkV2ZW50fSBldmVudCBUcmlnZ2VyZWQgZXZlbnQuXG4gICAgICAgICAqL1xuICAgICAgICBmdW5jdGlvbiBfb25VcGxvYWRJY29uQ2xpY2soZXZlbnQpIHtcbiAgICAgICAgICAgICQoZXZlbnQudGFyZ2V0KVxuICAgICAgICAgICAgICAgIC5wYXJlbnRzKHNlbGVjdG9ycy5jb25maWd1cmF0aW9uUm93KVxuICAgICAgICAgICAgICAgIC5maW5kKHNlbGVjdG9ycy5pbnB1dHMuZmlsZSlcbiAgICAgICAgICAgICAgICAudHJpZ2dlcignY2xpY2snKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiBIYW5kbGVzIHRoZSBmaWxlIHNlbGVjdCAoY2hhbmdlKSBldmVudC5cbiAgICAgICAgICpcbiAgICAgICAgICogQHBhcmFtIHtqUXVlcnkuRXZlbnR9IGV2ZW50IFRyaWdnZXJlZCBldmVudC5cbiAgICAgICAgICovXG4gICAgICAgIGZ1bmN0aW9uIF9vbkltYWdlQWRkKGV2ZW50KSB7XG4gICAgICAgICAgICAvLyBFeGl0IG1ldGhvZCwgaWYgc2VsZWN0aW9uIGhhcyBiZWVuIGFib3J0ZWQuXG4gICAgICAgICAgICBpZiAoIWV2ZW50LnRhcmdldC5maWxlcy5sZW5ndGgpIHtcbiAgICAgICAgICAgICAgICByZXR1cm47XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIC8vIEZpbGUgaW5wdXQgZWxlbWVudC5cbiAgICAgICAgICAgIGNvbnN0ICRmaWxlSW5wdXQgPSAkKGV2ZW50LnRhcmdldCk7XG5cbiAgICAgICAgICAgIC8vIEltYWdlIGRyb3Bkb3duLlxuICAgICAgICAgICAgY29uc3QgJGRyb3Bkb3duID0gJGZpbGVJbnB1dFxuICAgICAgICAgICAgICAgIC5wYXJlbnRzKHNlbGVjdG9ycy5jb25maWd1cmF0aW9uUm93KVxuICAgICAgICAgICAgICAgIC5maW5kKHNlbGVjdG9ycy5pbnB1dHMuZHJvcGRvd24pO1xuXG4gICAgICAgICAgICAvLyBSZWd1bGFyIGV4cHJlc3Npb24gdG8gdmFsaWRhdGUgdGhlIGZpbGUgbmFtZS5cbiAgICAgICAgICAgIGNvbnN0IHJlZ2V4ID0gLyguKShqcGd8anBlZ3xwbmd8Z2lmfGJtcCkkL2k7XG5cbiAgICAgICAgICAgIC8vIEZpbGUgbmFtZS5cbiAgICAgICAgICAgIGNvbnN0IGZpbGVOYW1lID0gZXZlbnQudGFyZ2V0LmZpbGVzWzBdLm5hbWU7XG5cbiAgICAgICAgICAgIC8vIElzIHRoZSBkcm9wZG93biBmb3IgdGh1bWJuYWlsIGltYWdlcz9cbiAgICAgICAgICAgIGNvbnN0IGlzVGh1bWJuYWlsSW1hZ2UgPSAhISRmaWxlSW5wdXRcbiAgICAgICAgICAgICAgICAucGFyZW50cyhzZWxlY3RvcnMuY29uZmlndXJhdGlvblJvdylcbiAgICAgICAgICAgICAgICAuZmluZChzZWxlY3RvcnMuaW5wdXRzLnRodW1ibmFpbEltYWdlRHJvcGRvd24pXG4gICAgICAgICAgICAgICAgLmxlbmd0aDtcblxuICAgICAgICAgICAgLy8gRXhpdCBtZXRob2QgYW5kIHNob3cgbW9kYWwsIGlmIGZpbGUgdHlwZSBkb2VzIG5vdCBtYXRjaC5cbiAgICAgICAgICAgIGlmICghcmVnZXgudGVzdChmaWxlTmFtZSkpIHtcbiAgICAgICAgICAgICAgICAvLyBTaG93IG1vZGFsLlxuICAgICAgICAgICAgICAgIGpzZS5saWJzLm1vZGFsLnNob3dNZXNzYWdlKFxuICAgICAgICAgICAgICAgICAgICBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnSU5WQUxJRF9GSUxFX01PREFMX1RJVExFJywgJ3NsaWRlcnMnKSxcbiAgICAgICAgICAgICAgICAgICAganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ0lOVkFMSURfRklMRV9NT0RBTF9URVhUJywgJ3NsaWRlcnMnKVxuICAgICAgICAgICAgICAgICk7XG5cbiAgICAgICAgICAgICAgICAvLyBSZXNldCB2YWx1ZS5cbiAgICAgICAgICAgICAgICAkZmlsZUlucHV0LnZhbCgnJyk7XG5cbiAgICAgICAgICAgICAgICByZXR1cm47XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIC8vIEV4aXQgbWV0aG9kIGFuZCBzaG93IG1vZGFsLCBpZiBmaWxlbmFtZSBpcyBhbHJlYWR5IHByZXNlbnQgaW4gZHJvcGRvd24uXG4gICAgICAgICAgICBmb3IgKGNvbnN0ICRvcHRpb24gb2YgJGRyb3Bkb3duWzBdLmNoaWxkcmVuKSB7XG4gICAgICAgICAgICAgICAgLy8gQ2hlY2sgaWYgb3B0aW9uJ3MgdGV4dCBjb250ZW50IG1hdGNoZXMgd2l0aCB0aGUgbmFtZSBvZiB0aGUgc2VsZWN0ZWQgZmlsZS5cbiAgICAgICAgICAgICAgICBpZiAoJG9wdGlvbi50ZXh0Q29udGVudCA9PT0gZmlsZU5hbWUpIHtcbiAgICAgICAgICAgICAgICAgICAgLy8gU2hvdyBtb2RhbC5cbiAgICAgICAgICAgICAgICAgICAganNlLmxpYnMubW9kYWwuc2hvd01lc3NhZ2UoXG4gICAgICAgICAgICAgICAgICAgICAgICBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnRklMRU5BTUVfQUxSRUFEWV9VU0VEX01PREFMX1RJVExFJywgJ3NsaWRlcnMnKSxcbiAgICAgICAgICAgICAgICAgICAgICAgIGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdGSUxFTkFNRV9BTFJFQURZX1VTRURfTU9EQUxfVEVYVCcsICdzbGlkZXJzJylcbiAgICAgICAgICAgICAgICAgICAgKTtcblxuICAgICAgICAgICAgICAgICAgICAvLyBSZXNldCB2YWx1ZS5cbiAgICAgICAgICAgICAgICAgICAgJGZpbGVJbnB1dC52YWwoJycpO1xuXG4gICAgICAgICAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIC8vIEFkZCBmaWxlcyB0byBkcm9wZG93bnMuXG4gICAgICAgICAgICBfYWRkSW1hZ2VUb0Ryb3Bkb3ducyhmaWxlTmFtZSwgaXNUaHVtYm5haWxJbWFnZSk7XG5cbiAgICAgICAgICAgIC8vIFNlbGVjdCB2YWx1ZS5cbiAgICAgICAgICAgICRkcm9wZG93blxuICAgICAgICAgICAgICAgIC52YWwoZmlsZU5hbWUpXG4gICAgICAgICAgICAgICAgLnRyaWdnZXIoJ2NoYW5nZScpO1xuICAgICAgICB9XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIEhhbmRsZXMgdGhlIGltYWdlIGRlbGV0ZSBidXR0b24gY2xpY2sgZXZlbnQuXG4gICAgICAgICAqXG4gICAgICAgICAqIEBwYXJhbSB7alF1ZXJ5LkV2ZW50fSBldmVudCBUcmlnZ2VyZWQgZXZlbnQuXG4gICAgICAgICAqL1xuICAgICAgICBmdW5jdGlvbiBfb25JbWFnZURlbGV0ZShldmVudCkge1xuICAgICAgICAgICAgLy8gTG9hZGluZyBDU1MgY2xhc3MgbmFtZS5cbiAgICAgICAgICAgIGNvbnN0IGxvYWRpbmdDbGFzcyA9ICdsb2FkaW5nJztcblxuICAgICAgICAgICAgLy8gSW1hZ2UgZHJvcGRvd24gY29udGFpbmVyLlxuICAgICAgICAgICAgY29uc3QgJGNvbmZpZ3VyYXRpb25Sb3cgPSAkKGV2ZW50LnRhcmdldCkucGFyZW50cyhzZWxlY3RvcnMuY29uZmlndXJhdGlvblJvdyk7XG5cbiAgICAgICAgICAgIC8vIEltYWdlIGRyb3Bkb3duLlxuICAgICAgICAgICAgY29uc3QgJGRyb3Bkb3duID0gJGNvbmZpZ3VyYXRpb25Sb3cuZmluZChzZWxlY3RvcnMuaW5wdXRzLmRyb3Bkb3duKTtcblxuICAgICAgICAgICAgLy8gU2xpZGUgSUQuXG4gICAgICAgICAgICBjb25zdCBzbGlkZUlkID0gJGNvbmZpZ3VyYXRpb25Sb3cucGFyZW50cyhzZWxlY3RvcnMuc2xpZGVQYW5lbCkuZGF0YSgnaWQnKTtcblxuICAgICAgICAgICAgLy8gU2xpZGUgaW1hZ2UgSUQuXG4gICAgICAgICAgICBjb25zdCBzbGlkZUltYWdlSWQgPSAkY29uZmlndXJhdGlvblJvdy5kYXRhKCdpZCcpO1xuXG4gICAgICAgICAgICAvLyBJcyB0aGUgZHJvcGRvd24gZm9yIHRodW1ibmFpbCBpbWFnZXM/XG4gICAgICAgICAgICBjb25zdCBpc1RodW1ibmFpbEltYWdlID0gISEkZHJvcGRvd24uaXMoc2VsZWN0b3JzLmlucHV0cy50aHVtYm5haWxJbWFnZURyb3Bkb3duKTtcblxuICAgICAgICAgICAgLy8gU2VsZWN0ZWQgZmlsZSBuYW1lLlxuICAgICAgICAgICAgY29uc3QgZmlsZU5hbWUgPSAkZHJvcGRvd24udmFsKCk7XG5cbiAgICAgICAgICAgIC8vIEFkZCBsb2FkaW5nIHN0YXRlLlxuICAgICAgICAgICAgJGRyb3Bkb3duLmFkZENsYXNzKGxvYWRpbmdDbGFzcyk7XG5cbiAgICAgICAgICAgIC8vIEltYWdlIHVzYWdlIGNoZWNrIHJlcXVlc3Qgb3B0aW9ucy5cbiAgICAgICAgICAgIGNvbnN0IHJlcXVlc3RPcHRpb25zID0ge1xuICAgICAgICAgICAgICAgIHVybDogJ2FkbWluLnBocD9kbz1TbGlkZXJzRGV0YWlsc0FqYXgvQ2hlY2tJbWFnZVVzYWdlJyxcbiAgICAgICAgICAgICAgICBkYXRhOiB7XG4gICAgICAgICAgICAgICAgICAgIGZpbGVuYW1lOiBmaWxlTmFtZSxcbiAgICAgICAgICAgICAgICAgICAgaXNfdGh1bWJuYWlsOiBpc1RodW1ibmFpbEltYWdlLFxuICAgICAgICAgICAgICAgICAgICBzbGlkZV9pZDogc2xpZGVJZCxcbiAgICAgICAgICAgICAgICAgICAgc2xpZGVfaW1hZ2VfaWQ6IHNsaWRlSW1hZ2VJZFxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH07XG5cbiAgICAgICAgICAgIC8vIFBlcmZvcm0gZGVsZXRpb24uXG4gICAgICAgICAgICBjb25zdCBwZXJmb3JtRGVsZXRpb24gPSAoKSA9PiB7XG4gICAgICAgICAgICAgICAgLy8gUHV0IGltYWdlIG5hbWUgaW50byBkZWxldGVyIHNlbGVjdCBib3guXG4gICAgICAgICAgICAgICAgZWxlbWVudHMuZGVsZXRlSW1hZ2VTZWxlY3QuYXBwZW5kKCQoJzxvcHRpb24+Jywge1xuICAgICAgICAgICAgICAgICAgICB2YWw6IGZpbGVOYW1lLFxuICAgICAgICAgICAgICAgICAgICBjbGFzczogaXNUaHVtYm5haWxJbWFnZSA/ICd0aHVtYm5haWwnIDogJydcbiAgICAgICAgICAgICAgICB9KSk7XG5cbiAgICAgICAgICAgICAgICAvLyBEZWxldGUgaW1hZ2UgZnJvbSBkcm9wZG93bnMuXG4gICAgICAgICAgICAgICAgX2RlbGV0ZUltYWdlRnJvbURyb3Bkb3ducyhmaWxlTmFtZSwgaXNUaHVtYm5haWxJbWFnZSk7XG4gICAgICAgICAgICB9O1xuXG4gICAgICAgICAgICAvLyBDaGVjayBpbWFnZSB1c2FnZS5cbiAgICAgICAgICAgIGpzZS5saWJzLnhoci5nZXQocmVxdWVzdE9wdGlvbnMpLnRoZW4ocmVzcG9uc2UgPT4ge1xuICAgICAgICAgICAgICAgIC8vIFJlbW92ZSBsb2FkaW5nIHN0YXRlLlxuICAgICAgICAgICAgICAgICRkcm9wZG93bi5yZW1vdmVDbGFzcyhsb2FkaW5nQ2xhc3MpO1xuXG4gICAgICAgICAgICAgICAgaWYgKHJlc3BvbnNlLmlzVXNlZCkge1xuICAgICAgICAgICAgICAgICAgICAvLyBNb2RhbCBjb25maXJtYXRpb24gYnV0dG9uIGVsZW1lbnQuXG4gICAgICAgICAgICAgICAgICAgIGNvbnN0ICRjb25maXJtQnV0dG9uID0gZWxlbWVudHMubW9kYWxzLmRlbGV0ZUltYWdlLmZpbmQoJ2J1dHRvbi5jb25maXJtJyk7XG5cbiAgICAgICAgICAgICAgICAgICAgLy8gU2hvdyBtb2RhbC5cbiAgICAgICAgICAgICAgICAgICAgZWxlbWVudHMubW9kYWxzLmRlbGV0ZUltYWdlLm1vZGFsKCdzaG93Jyk7XG5cbiAgICAgICAgICAgICAgICAgICAgLy8gTGlzdGVuIHRvIGNvbmZpcm1hdGlvbiBidXR0b24gY2xpY2sgZXZlbnQuXG4gICAgICAgICAgICAgICAgICAgICRjb25maXJtQnV0dG9uXG4gICAgICAgICAgICAgICAgICAgICAgICAub2ZmKCdjbGljaycpXG4gICAgICAgICAgICAgICAgICAgICAgICAub24oJ2NsaWNrJywgcGVyZm9ybURlbGV0aW9uKTtcbiAgICAgICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICAgICAgICBwZXJmb3JtRGVsZXRpb24oKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiBIYW5kbGVzIHRoZSBpbWFnZSBtYXAgZWRpdCBidXR0b24gY2xpY2sgZXZlbnQuXG4gICAgICAgICAqXG4gICAgICAgICAqIEBwYXJhbSB7alF1ZXJ5LkV2ZW50fSBldmVudCBUcmlnZ2VyZWQgZXZlbnQuXG4gICAgICAgICAqL1xuICAgICAgICBmdW5jdGlvbiBfb25JbWFnZU1hcChldmVudCkge1xuICAgICAgICAgICAgLy8gU2xpZGUgaW1hZ2UgSUQuXG4gICAgICAgICAgICBjb25zdCBzbGlkZUltYWdlSWQgPSAkKGV2ZW50LnRhcmdldClcbiAgICAgICAgICAgICAgICAucGFyZW50cyhzZWxlY3RvcnMuY29uZmlndXJhdGlvblJvdylcbiAgICAgICAgICAgICAgICAuZGF0YSgnaWQnKTtcblxuICAgICAgICAgICAgLy8gTGlzdCBlbGVtZW50IHdoaWNoIGFjdHMgbGlrZSBhIGRhdGEgY29udGFpbmVyLlxuICAgICAgICAgICAgY29uc3QgJGxpc3QgPSAkKGV2ZW50LnRhcmdldClcbiAgICAgICAgICAgICAgICAucGFyZW50cyhzZWxlY3RvcnMuY29uZmlndXJhdGlvblJvdylcbiAgICAgICAgICAgICAgICAuZmluZChzZWxlY3RvcnMuaW1hZ2VNYXBEYXRhTGlzdCk7XG5cbiAgICAgICAgICAgIC8vIEltYWdlIGRyb3Bkb3duLlxuICAgICAgICAgICAgY29uc3QgJGRyb3Bkb3duID0gJChldmVudC50YXJnZXQpXG4gICAgICAgICAgICAgICAgLnBhcmVudHMoc2VsZWN0b3JzLmNvbmZpZ3VyYXRpb25Sb3cpXG4gICAgICAgICAgICAgICAgLmZpbmQoc2VsZWN0b3JzLmlucHV0cy5kcm9wZG93bik7XG5cbiAgICAgICAgICAgIC8vIFNsaWRlIGltYWdlIGZpbGUgbmFtZS5cbiAgICAgICAgICAgIGNvbnN0IGltYWdlRmlsZW5hbWUgPSAkZHJvcGRvd24udmFsKCk7XG5cbiAgICAgICAgICAgIC8vIElzIGEgbmV3IGltYWdlIHNlbGVjdGVkP1xuICAgICAgICAgICAgY29uc3QgaXNOZXdJbWFnZVNlbGVjdGVkID0gJGRyb3Bkb3duLmZpbmQoJ29wdGlvbjpzZWxlY3RlZCcpLmhhc0NsYXNzKGNsYXNzZXMubmV3SW1hZ2UpO1xuXG4gICAgICAgICAgICAvLyBQYXRoIHRvIGltYWdlIFVSTC5cbiAgICAgICAgICAgIGNvbnN0IGltYWdlVXJsID0ganNlLmNvcmUuY29uZmlnLmdldCgnYXBwVXJsJykgKyAnL2ltYWdlcy9zbGlkZXJfaW1hZ2VzLycgKyBpbWFnZUZpbGVuYW1lO1xuXG4gICAgICAgICAgICAvLyBTaG93IHNhdmUgZmlyc3Qgbm90aWNlIG1vZGFsIGFuZCByZXR1cm4gaW1tZWRpYXRlbHksIGlmIHRoZSBzbGlkZSBpbWFnZSBoYXMgbm8gSUQuXG4gICAgICAgICAgICBpZiAoIXNsaWRlSW1hZ2VJZCB8fCBpc05ld0ltYWdlU2VsZWN0ZWQpIHtcbiAgICAgICAgICAgICAgICBqc2UubGlicy5tb2RhbC5zaG93TWVzc2FnZShcbiAgICAgICAgICAgICAgICAgICAganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ0lNQUdFX01BUF9NT0RBTF9USVRMRScsICdzbGlkZXJzJyksXG4gICAgICAgICAgICAgICAgICAgIGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdTQVZFX1NMSURFUl9GSVJTVF9OT1RJQ0VfVEVYVCcsICdzbGlkZXJzJylcbiAgICAgICAgICAgICAgICApO1xuXG4gICAgICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICAvLyBTaG93IGltYWdlIG1hcCBtb2RhbC5cbiAgICAgICAgICAgIGVsZW1lbnRzLm1vZGFscy5pbWFnZU1hcC50cmlnZ2VyKCdzaG93JywgWyRsaXN0LCBpbWFnZVVybF0pO1xuICAgICAgICB9XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIEhhbmRsZXMgdGhlIHNvcnQgYnV0dG9uIGNsaWNrIGV2ZW50LlxuICAgICAgICAgKi9cbiAgICAgICAgZnVuY3Rpb24gX29uU29ydEJ1dHRvbkNsaWNrKCkge1xuICAgICAgICAgICAgLy8gSW5kaWNhdG9yIENTUyBjbGFzc2VzLlxuICAgICAgICAgICAgY29uc3QgaW5kaWNhdG9yQ2xhc3MgPSAnbW9kZS1vbiBidG4tcHJpbWFyeSc7XG5cbiAgICAgICAgICAgIC8vIFNlbGVjdG9yIHN0cmluZyBmb3IgdGhlIHNsaWRlIHBhbmVsIGJvZHkuXG4gICAgICAgICAgICBjb25zdCBzbGlkZVBhbmVsQm9keVNlbGVjdG9yID0gJy5wYW5lbC1ib2R5JztcblxuICAgICAgICAgICAgLy8gU2xpZGVzIGNvbnRhaW5lciB0YWJzLCBleGNlcHQgdGhlIGFjdGl2ZSBvbmUuXG4gICAgICAgICAgICBjb25zdCAkb3RoZXJUYWJzID0gZWxlbWVudHMudGFiSGVhZGVyXG4gICAgICAgICAgICAgICAgLmNoaWxkcmVuKClcbiAgICAgICAgICAgICAgICAubm90KCcuYWN0aXZlJyk7XG5cbiAgICAgICAgICAgIC8vIElzIHRoZSBzb3J0IG1vZGUgb24/XG4gICAgICAgICAgICBjb25zdCBpc01vZGVPbiA9IGVsZW1lbnRzLmJ1dHRvbnMuc29ydC5oYXNDbGFzcyhpbmRpY2F0b3JDbGFzcyk7XG5cbiAgICAgICAgICAgIC8vIExhbmd1YWdlLXNwZWNpZmljIGJ1dHRvbiB0ZXh0cy5cbiAgICAgICAgICAgIGNvbnN0IGVudGVyVGV4dCA9IGVsZW1lbnRzLmJ1dHRvbnMuc29ydC5kYXRhKCd0ZXh0RW50ZXInKTtcbiAgICAgICAgICAgIGNvbnN0IGV4aXRUZXh0ID0gZWxlbWVudHMuYnV0dG9ucy5zb3J0LmRhdGEoJ3RleHRFeGl0Jyk7XG5cbiAgICAgICAgICAgIC8vIEFsbCBzbGlkZSBwYW5lbHMuXG4gICAgICAgICAgICBjb25zdCAkc2xpZGVzID0gJHRoaXMuZmluZChzZWxlY3RvcnMuc2xpZGVQYW5lbCk7XG5cbiAgICAgICAgICAgIC8vIEFwcGx5IGZhZGUgZWZmZWN0IG9udG8gc2xpZGUgcGFuZWxzLlxuICAgICAgICAgICAgJHNsaWRlc1xuICAgICAgICAgICAgICAgIC5oaWRlKClcbiAgICAgICAgICAgICAgICAuZmFkZUluKCk7XG5cbiAgICAgICAgICAgIC8vIFN3aXRjaCB0ZXh0IGFuZCB0b2dnbGUgaW5kaWNhdG9yIGNsYXNzLlxuICAgICAgICAgICAgZWxlbWVudHMuYnV0dG9ucy5zb3J0W2lzTW9kZU9uID8gJ3JlbW92ZUNsYXNzJyA6ICdhZGRDbGFzcyddKGluZGljYXRvckNsYXNzKVxuICAgICAgICAgICAgICAgIC50ZXh0KGlzTW9kZU9uID8gZW50ZXJUZXh0IDogZXhpdFRleHQpO1xuXG4gICAgICAgICAgICAvLyBUb2dnbGUgY3JlYXRlIGJ1dHRvbi5cbiAgICAgICAgICAgIGVsZW1lbnRzLmJ1dHRvbnMuY3JlYXRlLnByb3AoJ2Rpc2FibGVkJywgIWlzTW9kZU9uKTtcblxuICAgICAgICAgICAgLy8gVG9nZ2xlIGRyYWcgaGFuZGxlIGJ1dHRvbnMuXG4gICAgICAgICAgICAkc2xpZGVzLmZpbmQoc2VsZWN0b3JzLmljb25zLmRyYWcpW2lzTW9kZU9uID8gJ2hpZGUnIDogJ3Nob3cnXSgpO1xuXG4gICAgICAgICAgICAvLyBUb2dnbGUgb3RoZXIgdGFicy5cbiAgICAgICAgICAgICRvdGhlclRhYnNbaXNNb2RlT24gPyAnZmFkZUluJyA6ICdmYWRlT3V0J10oKTtcblxuICAgICAgICAgICAgLy8gVG9nZ2xlIGNvbGxhcHNlciBhbmQgaGlkZSBidXR0b25zLlxuICAgICAgICAgICAgJHNsaWRlc1xuICAgICAgICAgICAgICAgIC5maW5kKHNlbGVjdG9ycy5pY29ucy5jb2xsYXBzZXIpXG4gICAgICAgICAgICAgICAgLmFkZChzZWxlY3RvcnMuaWNvbnMuZGVsZXRlKVtpc01vZGVPbiA/ICdzaG93JyA6ICdoaWRlJ10oKTtcblxuICAgICAgICAgICAgLy8gU2F2ZSBvcGVuIHNsaWRlIHBhbmVscy5cbiAgICAgICAgICAgIGlmICghaXNNb2RlT24pIHtcbiAgICAgICAgICAgICAgICBvcGVuU2xpZGVQYW5lbHMgPSAkc2xpZGVzLmZpbHRlcigoaW5kZXgsIGVsZW1lbnQpID0+ICQoZWxlbWVudClcbiAgICAgICAgICAgICAgICAgICAgLmZpbmQoc2xpZGVQYW5lbEJvZHlTZWxlY3RvcilcbiAgICAgICAgICAgICAgICAgICAgLmlzKCc6dmlzaWJsZScpKTtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgLy8gVG9nZ2xlIHNhdmVkIG9wZW4gc2xpZGUgcGFuZWxzLlxuICAgICAgICAgICAgb3BlblNsaWRlUGFuZWxzLmVhY2goKGluZGV4LCBlbGVtZW50KSA9PiAkKGVsZW1lbnQpXG4gICAgICAgICAgICAgICAgLmZpbmQoc2VsZWN0b3JzLmljb25zLmNvbGxhcHNlcilcbiAgICAgICAgICAgICAgICAudHJpZ2dlcignY2xpY2snKSk7XG4gICAgICAgIH1cblxuICAgICAgICAvKipcbiAgICAgICAgICogSGFuZGxlcyB0aGUgc29ydCBzdGFydCBldmVudC5cbiAgICAgICAgICovXG4gICAgICAgIGZ1bmN0aW9uIF9vblNvcnRTdGFydCgpIHtcbiAgICAgICAgICAgIC8vIFRhYiBjb250ZW50IGVsZW1lbnQgZm9yIHNlbGVjdGVkIGxhbmd1YWdlLlxuICAgICAgICAgICAgY29uc3QgJHRhYkJvZHkgPSAkdGhpc1xuICAgICAgICAgICAgICAgIC5maW5kKHNlbGVjdG9ycy50YWJCb2R5KVxuICAgICAgICAgICAgICAgIC5maWx0ZXIoJzp2aXNpYmxlJyk7XG5cbiAgICAgICAgICAgIC8vIFJlZnJlc2ggdGFiIHNpemVzIGFuZCBwb3NpdGlvbnMuXG4gICAgICAgICAgICAkdGFiQm9keS5zb3J0YWJsZSgncmVmcmVzaFBvc2l0aW9ucycpO1xuICAgICAgICB9XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIEhhbmRsZXMgdGhlIHNvcnQgc3RvcCBldmVudC5cbiAgICAgICAgICovXG4gICAgICAgIGZ1bmN0aW9uIF9vblNvcnRTdG9wKCkge1xuICAgICAgICAgICAgLy8gUmVnaXN0ZXIgY2hhbmdlLCB0byBtYWtlIHByb21wdCBvbiBwYWdlIHVubG9hZC5cbiAgICAgICAgICAgIF9yZWdpc3RlckNoYW5nZSgpO1xuICAgICAgICB9XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIEhhbmRsZXMgdGhlIGRlbGV0ZSBpY29uIGNsaWNrIGV2ZW50LlxuICAgICAgICAgKlxuICAgICAgICAgKiBAcGFyYW0ge2pRdWVyeS5FdmVudH0gZXZlbnQgVHJpZ2dlcmVkIGV2ZW50LlxuICAgICAgICAgKi9cbiAgICAgICAgZnVuY3Rpb24gX29uRGVsZXRlSWNvbkNsaWNrKGV2ZW50KSB7XG4gICAgICAgICAgICAvLyBTbGlkZSBwYW5lbCBlbGVtZW50LlxuICAgICAgICAgICAgY29uc3QgJHNsaWRlUGFuZWwgPSAkKGV2ZW50LnRhcmdldCkucGFyZW50cyhzZWxlY3RvcnMuc2xpZGVQYW5lbCk7XG5cbiAgICAgICAgICAgIC8vIE1vZGFsIGNvbmZpcm1hdGlvbiBidXR0b24gZWxlbWVudC5cbiAgICAgICAgICAgIGNvbnN0ICRjb25maXJtQnV0dG9uID0gZWxlbWVudHMubW9kYWxzLmRlbGV0ZVNsaWRlLmZpbmQoJ2J1dHRvbi5jb25maXJtJyk7XG5cbiAgICAgICAgICAgIC8vIFNob3cgbW9kYWwuXG4gICAgICAgICAgICBlbGVtZW50cy5tb2RhbHMuZGVsZXRlU2xpZGUubW9kYWwoJ3Nob3cnKTtcblxuICAgICAgICAgICAgLy8gTGlzdGVuIHRvIGNvbmZpcm1hdGlvbiBidXR0b24gY2xpY2sgZXZlbnQuXG4gICAgICAgICAgICAkY29uZmlybUJ1dHRvblxuICAgICAgICAgICAgICAgIC5vZmYoJ2NsaWNrJylcbiAgICAgICAgICAgICAgICAub24oJ2NsaWNrJywgKCkgPT4gX29uRGVsZXRlQ29uZmlybWF0aW9uQnV0dG9uQ2xpY2soZWxlbWVudHMubW9kYWxzLmRlbGV0ZVNsaWRlLCAkc2xpZGVQYW5lbCkpO1xuICAgICAgICB9XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIEhhbmRsZXMgdGhlIGNyZWF0ZSBidXR0b24gY2xpY2sgZXZlbnQuXG4gICAgICAgICAqL1xuICAgICAgICBmdW5jdGlvbiBfb25DcmVhdGVCdXR0b25DbGljaygpIHtcbiAgICAgICAgICAgIC8vIE1ha2UgYSBjbG9uZSBvZiB0aGUgc2xpZGUgcGFuZWwgdGVtcGxhdGUgYW5kIGNyZWF0ZSBhIG5ldyBlbGVtZW50LlxuICAgICAgICAgICAgY29uc3QgJHNsaWRlUGFuZWwgPSAkKGVsZW1lbnRzLnRlbXBsYXRlcy5zbGlkZVBhbmVsLmNsb25lKCkuaHRtbCgpKTtcblxuICAgICAgICAgICAgLy8gVGFiIGNvbnRlbnQgZWxlbWVudCBmb3Igc2VsZWN0ZWQgbGFuZ3VhZ2UuXG4gICAgICAgICAgICBjb25zdCAkdGFiQm9keSA9ICR0aGlzXG4gICAgICAgICAgICAgICAgLmZpbmQoc2VsZWN0b3JzLnRhYkJvZHkpXG4gICAgICAgICAgICAgICAgLmZpbHRlcignOnZpc2libGUnKTtcblxuICAgICAgICAgICAgLy8gU2xpZGUgcGFuZWxzLlxuICAgICAgICAgICAgY29uc3QgJHNsaWRlUGFuZWxzID0gJHRhYkJvZHkuZmluZChzZWxlY3RvcnMuc2xpZGVQYW5lbCk7XG5cbiAgICAgICAgICAgIC8vIE5leHQgcGFuZWwgaW5kZXguXG4gICAgICAgICAgICBjb25zdCBwYW5lbEluZGV4ID0gJHNsaWRlUGFuZWxzLmxlbmd0aCArIDE7XG5cbiAgICAgICAgICAgIC8vIFRpdGxlIGZvciBuZXcgc2xpZGUgcGFuZWwuXG4gICAgICAgICAgICBjb25zdCBuZXdUaXRsZSA9IGAke2pzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdORVdfU0xJREUnLCAnc2xpZGVycycpfSAke3BhbmVsSW5kZXh9YDtcblxuICAgICAgICAgICAgLy8gQWRkIHRpdGxlIHRvIHNsaWRlIHBhbmVsIGhlYWRlci5cbiAgICAgICAgICAgICRzbGlkZVBhbmVsXG4gICAgICAgICAgICAgICAgLmZpbmQoc2VsZWN0b3JzLnNsaWRlUGFuZWxUaXRsZSlcbiAgICAgICAgICAgICAgICAudGV4dChuZXdUaXRsZSk7XG5cbiAgICAgICAgICAgIC8vIEFkZCB0aXRsZSB0byBpbnB1dCBmaWVsZC5cbiAgICAgICAgICAgICRzbGlkZVBhbmVsXG4gICAgICAgICAgICAgICAgLmZpbmQoc2VsZWN0b3JzLmlucHV0cy50aXRsZSlcbiAgICAgICAgICAgICAgICAudmFsKG5ld1RpdGxlKTtcblxuICAgICAgICAgICAgLy8gQWRkIHZhbHVlcyB0byBkcm9wZG93bnMuXG4gICAgICAgICAgICBpZiAoJHNsaWRlUGFuZWxzLmxlbmd0aCkge1xuICAgICAgICAgICAgICAgIC8vIEdldCBhbGwgaW1hZ2UgZHJvcGRvd25zIG9mIHRoZSBmaXJzdCBwYW5lbC5cbiAgICAgICAgICAgICAgICBjb25zdCAkZHJvcGRvd25zID0gJHNsaWRlUGFuZWxzXG4gICAgICAgICAgICAgICAgICAgIC5maXJzdCgpXG4gICAgICAgICAgICAgICAgICAgIC5maW5kKHNlbGVjdG9ycy5pbnB1dHMuZHJvcGRvd24pO1xuXG4gICAgICAgICAgICAgICAgLy8gR2V0IHRoZSB0aHVtYm5haWwgZHJvcGRvd24gb3B0aW9ucy5cbiAgICAgICAgICAgICAgICBjb25zdCAkdGh1bWJuYWlsT3B0aW9ucyA9ICRkcm9wZG93bnNcbiAgICAgICAgICAgICAgICAgICAgLmZpbHRlcihzZWxlY3RvcnMuaW5wdXRzLnRodW1ibmFpbEltYWdlRHJvcGRvd24pXG4gICAgICAgICAgICAgICAgICAgIC5jaGlsZHJlbigpXG4gICAgICAgICAgICAgICAgICAgIC5jbG9uZSgpO1xuXG4gICAgICAgICAgICAgICAgLy8gR2V0IHRoZSBpbWFnZSBkcm9wZG93biBvcHRpb25zLlxuICAgICAgICAgICAgICAgIGNvbnN0ICRpbWFnZU9wdGlvbnMgPSAkZHJvcGRvd25zXG4gICAgICAgICAgICAgICAgICAgIC5ub3Qoc2VsZWN0b3JzLmlucHV0cy50aHVtYm5haWxJbWFnZURyb3Bkb3duKVxuICAgICAgICAgICAgICAgICAgICAuZmlyc3QoKVxuICAgICAgICAgICAgICAgICAgICAuY2hpbGRyZW4oKVxuICAgICAgICAgICAgICAgICAgICAuY2xvbmUoKTtcblxuICAgICAgICAgICAgICAgIC8vIFJlcGxhY2UgdGh1bWJuYWlsIG9wdGlvbnMgaW4gbmV3IHNsaWRlIHBhbmVsLlxuICAgICAgICAgICAgICAgICRzbGlkZVBhbmVsXG4gICAgICAgICAgICAgICAgICAgIC5maW5kKHNlbGVjdG9ycy5pbnB1dHMudGh1bWJuYWlsSW1hZ2VEcm9wZG93bilcbiAgICAgICAgICAgICAgICAgICAgLmVtcHR5KClcbiAgICAgICAgICAgICAgICAgICAgLmFwcGVuZCgkdGh1bWJuYWlsT3B0aW9ucylcbiAgICAgICAgICAgICAgICAgICAgLnZhbCgnJyk7XG5cbiAgICAgICAgICAgICAgICAvLyBSZXBsYWNlIGltYWdlIG9wdGlvbnMgaW4gbmV3IHNsaWRlIHBhbmVsLlxuICAgICAgICAgICAgICAgICRzbGlkZVBhbmVsXG4gICAgICAgICAgICAgICAgICAgIC5maW5kKHNlbGVjdG9ycy5pbnB1dHMuZHJvcGRvd24pXG4gICAgICAgICAgICAgICAgICAgIC5ub3Qoc2VsZWN0b3JzLmlucHV0cy50aHVtYm5haWxJbWFnZURyb3Bkb3duKVxuICAgICAgICAgICAgICAgICAgICAuZW1wdHkoKVxuICAgICAgICAgICAgICAgICAgICAuYXBwZW5kKCRpbWFnZU9wdGlvbnMpXG4gICAgICAgICAgICAgICAgICAgIC52YWwoJycpO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICAvLyBBZGQgbmV3IHNsaWRlIHBhbmVsIGVsZW1lbnQgdG8gdGFiIGJvZHkgd2l0aCBmYWRlIGVmZmVjdC5cbiAgICAgICAgICAgICRzbGlkZVBhbmVsXG4gICAgICAgICAgICAgICAgLmhpZGUoKVxuICAgICAgICAgICAgICAgIC5wcmVwZW5kVG8oJHRhYkJvZHkpXG4gICAgICAgICAgICAgICAgLmZhZGVJbigpO1xuXG4gICAgICAgICAgICAvLyBJbml0aWFsaXplIHdpZGdldHMgYW5kIGV4dGVuc2lvbnMgb24gdGhlIG5ldyBzbGlkZSBwYW5lbCBlbGVtZW50LlxuICAgICAgICAgICAgZ3gud2lkZ2V0cy5pbml0KCRzbGlkZVBhbmVsKTtcbiAgICAgICAgICAgIGd4LmV4dGVuc2lvbnMuaW5pdCgkc2xpZGVQYW5lbCk7XG5cbiAgICAgICAgICAgIC8vIFRyaWdnZXIgY2hhbmdlIHRvIHNob3cgdGhlIHJpZ2h0IGFjdGlvbiBpY29ucy5cbiAgICAgICAgICAgICRzbGlkZVBhbmVsXG4gICAgICAgICAgICAgICAgLmZpbmQoJ3NlbGVjdCcpXG4gICAgICAgICAgICAgICAgLnRyaWdnZXIoJ2NoYW5nZScpO1xuXG4gICAgICAgICAgICAvLyBSZWdpc3RlciBjaGFuZ2UsIHRvIG1ha2UgcHJvbXB0IG9uIHBhZ2UgdW5sb2FkLlxuICAgICAgICAgICAgX3JlZ2lzdGVyQ2hhbmdlKCk7XG5cbiAgICAgICAgICAgIC8vIFRvZ2dsZSBzdWJtaXQgYnV0dG9ucy5cbiAgICAgICAgICAgIHRvZ2dsZVN1Ym1pdEJ1dHRvbnMoKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiBIYW5kbGVzIHRoZSBjb25maXJtYXRpb24gYnV0dG9uIGNsaWNrIGV2ZW50IGluIHRoZSBkZWxldGUgY29uZmlybWF0aW9uIG1vZGFsLlxuICAgICAgICAgKlxuICAgICAgICAgKiBAcGFyYW0ge2pRdWVyeX0gJG1vZGFsIERlbGV0ZSBjb25maXJtYXRpb24gbW9kYWwgZWxlbWVudC5cbiAgICAgICAgICogQHBhcmFtIHtqUXVlcnl9ICRzbGlkZVBhbmVsIFNsaWRlIHBhbmVsIGVsZW1lbnQuXG4gICAgICAgICAqL1xuICAgICAgICBmdW5jdGlvbiBfb25EZWxldGVDb25maXJtYXRpb25CdXR0b25DbGljaygkbW9kYWwsICRzbGlkZVBhbmVsKSB7XG4gICAgICAgICAgICAvLyBIaWRlIG1vZGFsLlxuICAgICAgICAgICAgJG1vZGFsLm1vZGFsKCdoaWRlJyk7XG5cbiAgICAgICAgICAgIC8vIEZhZGUgb3V0IHNsaWRlIHBhbmVsIGVsZW1lbnQgYW5kIHRoZW4gcmVtb3ZlIGl0LlxuICAgICAgICAgICAgJHNsaWRlUGFuZWwuZmFkZU91dCg0MDAsICgpID0+IHtcbiAgICAgICAgICAgICAgICAvLyBSZW1vdmUgc2xpZGUgcGFuZWwuXG4gICAgICAgICAgICAgICAgJHNsaWRlUGFuZWwucmVtb3ZlKClcblxuICAgICAgICAgICAgICAgIC8vIFRvZ2dsZSBzdWJtaXQgYnV0dG9ucy5cbiAgICAgICAgICAgICAgICB0b2dnbGVTdWJtaXRCdXR0b25zKCk7XG4gICAgICAgICAgICB9KTtcblxuICAgICAgICAgICAgLy8gUmVnaXN0ZXIgY2hhbmdlLCB0byBtYWtlIHByb21wdCBvbiBwYWdlIHVubG9hZC5cbiAgICAgICAgICAgIF9yZWdpc3RlckNoYW5nZSgpO1xuICAgICAgICB9XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIEhhbmRsZXMgdGhlIGtleS11cCBldmVudCBvbiB0aGUgc2xpZGUgdGl0bGUgaW5wdXQgZmllbGQuXG4gICAgICAgICAqXG4gICAgICAgICAqIEBwYXJhbSB7alF1ZXJ5LkV2ZW50fSBldmVudCBUcmlnZ2VyZWQgZXZlbnQuXG4gICAgICAgICAqL1xuICAgICAgICBmdW5jdGlvbiBfb25UaXRsZUtleXVwKGV2ZW50KSB7XG4gICAgICAgICAgICAvLyBUaXRsZSBpbnB1dCBmaWVsZC5cbiAgICAgICAgICAgIGNvbnN0ICRpbnB1dCA9ICQoZXZlbnQudGFyZ2V0KTtcblxuICAgICAgICAgICAgLy8gU2xpZGUgcGFuZWwgdGl0bGUgZWxlbWVudC5cbiAgICAgICAgICAgIGNvbnN0ICR0aXRsZSA9ICRpbnB1dFxuICAgICAgICAgICAgICAgIC5wYXJlbnRzKHNlbGVjdG9ycy5zbGlkZVBhbmVsKVxuICAgICAgICAgICAgICAgIC5maW5kKHNlbGVjdG9ycy5zbGlkZVBhbmVsVGl0bGUpO1xuXG4gICAgICAgICAgICAvLyBUcmFuc2ZlciBpbnB1dCB2YWx1ZSB0byBzbGlkZSBwYW5lbCB0aXRsZS5cbiAgICAgICAgICAgICR0aXRsZS50ZXh0KCRpbnB1dC52YWwoKSk7XG4gICAgICAgIH1cblxuICAgICAgICAvKipcbiAgICAgICAgICogSGFuZGxlcyB0aGUgbW91c2UtZW50ZXIgZXZlbnQgb24gYSBjb25maWd1cmF0aW9uIHJvdy5cbiAgICAgICAgICpcbiAgICAgICAgICogQHBhcmFtIHtqUXVlcnkuRXZlbnR9IGV2ZW50IFRyaWdnZXJlZCBldmVudC5cbiAgICAgICAgICovXG4gICAgICAgIGZ1bmN0aW9uIF9vbkNvbmZpZ1Jvd01vdXNlRW50ZXIoZXZlbnQpIHtcbiAgICAgICAgICAgIC8vIENvbmZpZ3VyYXRpb24gcm93IGVsZW1lbnQuXG4gICAgICAgICAgICBjb25zdCAkcm93ID0gJChldmVudC50YXJnZXQpO1xuXG4gICAgICAgICAgICAvLyBJbWFnZSBtYXAgZWRpdCBpY29uLlxuICAgICAgICAgICAgY29uc3QgJGltYWdlTWFwSWNvbiA9ICRyb3cuZmluZChzZWxlY3RvcnMuaWNvbnMuaW1hZ2VNYXApO1xuXG4gICAgICAgICAgICAvLyBJbWFnZSBtYXAgZGF0YSBjb250YWluZXIgbGlzdCBlbGVtZW50LlxuICAgICAgICAgICAgY29uc3QgJGxpc3QgPSAkcm93LmZpbmQoc2VsZWN0b3JzLmltYWdlTWFwRGF0YUxpc3QpO1xuXG4gICAgICAgICAgICAvLyBSZXR1cm4gaW1tZWRpYXRlbHksIGlmIHRoZSBpbWFnZSBtYXAgZWRpdCBpY29uIGRvZXMgbm90IGV4aXN0LlxuICAgICAgICAgICAgaWYgKCEkaW1hZ2VNYXBJY29uLmxlbmd0aCB8fCAhJGxpc3QubGVuZ3RoKSB7XG4gICAgICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICBpZiAoJGxpc3QuY2hpbGRyZW4oKS5sZW5ndGgpIHtcbiAgICAgICAgICAgICAgICAkaW1hZ2VNYXBJY29uLnJlbW92ZUNsYXNzKCdmYS1leHRlcm5hbC1saW5rJykuYWRkQ2xhc3MoJ2ZhLWV4dGVybmFsLWxpbmstc3F1YXJlJyk7XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICAgICRpbWFnZU1hcEljb24ucmVtb3ZlQ2xhc3MoJ2ZhLWV4dGVybmFsLWxpbmstc3F1YXJlJykuYWRkQ2xhc3MoJ2ZhLWV4dGVybmFsLWxpbmsnKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiBIYW5kbGVzIHRoZSBjbGljayBldmVudCBvbiB0aGUgc2F2ZSBidXR0b24uXG4gICAgICAgICAqL1xuICAgICAgICBmdW5jdGlvbiBfb25TdWJtaXRTYXZlKCkge1xuICAgICAgICAgICAgJHRoaXNcbiAgICAgICAgICAgICAgICAucGFyZW50cygnZm9ybScpXG4gICAgICAgICAgICAgICAgLnRyaWdnZXIoJ3N1Ym1pdCcpO1xuICAgICAgICB9XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIEhhbmRsZXMgdGhlIGNsaWNrIGV2ZW50IG9uIHRoZSByZWZyZXNoIGxpc3QgaXRlbSBpbiB0aGUgc3VibWl0IGJ1dHRvbiBncm91cC5cbiAgICAgICAgICovXG4gICAgICAgIGZ1bmN0aW9uIF9vblN1Ym1pdFJlZnJlc2goKSB7XG4gICAgICAgICAgICAkdGhpc1xuICAgICAgICAgICAgICAgIC5wYXJlbnRzKCdmb3JtJylcbiAgICAgICAgICAgICAgICAudHJpZ2dlcignc3VibWl0Jywge3JlZnJlc2g6IHRydWV9KTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8qKlxuICAgICAgICAgKiBBZGRzIGFuIGltYWdlIHRvIHRoZSBpbWFnZSBkcm9wZG93bnMuXG4gICAgICAgICAqXG4gICAgICAgICAqIEBwYXJhbSB7U3RyaW5nfSBmaWxlTmFtZSBOYW1lIG9mIHRoZSBzZWxlY3RlZCBmaWxlLlxuICAgICAgICAgKiBAcGFyYW0ge0Jvb2xlYW59IFt0aHVtYm5haWxJbWFnZXNPbmx5ID0gZmFsc2VdIEFwcGx5IG9uIHRodW1ibmFpbCBpbWFnZSBkcm9wZG93bnMgb25seT9cbiAgICAgICAgICovXG4gICAgICAgIGZ1bmN0aW9uIF9hZGRJbWFnZVRvRHJvcGRvd25zKGZpbGVOYW1lLCB0aHVtYm5haWxJbWFnZXNPbmx5ID0gZmFsc2UpIHtcbiAgICAgICAgICAgIC8vIFNlbGVjdCBzcGVjaWZpYyBkcm9wZG93bnMuXG4gICAgICAgICAgICBjb25zdCAkZHJvcGRvd25zID0gJHRoaXNcbiAgICAgICAgICAgICAgICAuZmluZChzZWxlY3RvcnMuaW5wdXRzLmRyb3Bkb3duKVxuICAgICAgICAgICAgICAgIFt0aHVtYm5haWxJbWFnZXNPbmx5ID8gJ2ZpbHRlcicgOiAnbm90J10oc2VsZWN0b3JzLmlucHV0cy50aHVtYm5haWxJbWFnZURyb3Bkb3duKTtcblxuICAgICAgICAgICAgLy8gQ3JlYXRlIG5ldyBpbWFnZSBvcHRpb24gZWxlbWVudC5cbiAgICAgICAgICAgIGNvbnN0ICRvcHRpb24gPSAkKCc8b3B0aW9uPicsIHt2YWx1ZTogZmlsZU5hbWUsIHRleHQ6IGZpbGVOYW1lLCBjbGFzczogY2xhc3Nlcy5uZXdJbWFnZX0pO1xuXG4gICAgICAgICAgICAvLyBBcHBlbmQgbmV3IG9wdGlvbnMgdG8gZHJvcGRvd25zLlxuICAgICAgICAgICAgJGRyb3Bkb3ducy5hcHBlbmQoJG9wdGlvbik7XG4gICAgICAgIH1cblxuICAgICAgICAvKipcbiAgICAgICAgICogRGVsZXRlcyBhbiBpbWFnZSBmcm9tIHRoZSBpbWFnZSBkcm9wZG93bnMuXG4gICAgICAgICAqXG4gICAgICAgICAqIEBwYXJhbSB7U3RyaW5nfSBmaWxlTmFtZSBOYW1lIG9mIHRoZSBzZWxlY3RlZCBmaWxlLlxuICAgICAgICAgKiBAcGFyYW0ge0Jvb2xlYW59IFt0aHVtYm5haWxJbWFnZXNPbmx5ID0gZmFsc2VdIEFwcGx5IG9uIHRodW1ibmFpbCBpbWFnZSBkcm9wZG93bnMgb25seT9cbiAgICAgICAgICovXG4gICAgICAgIGZ1bmN0aW9uIF9kZWxldGVJbWFnZUZyb21Ecm9wZG93bnMoZmlsZU5hbWUsIHRodW1ibmFpbEltYWdlc09ubHkgPSBmYWxzZSkge1xuICAgICAgICAgICAgLy8gU2VsZWN0IGFsbCBkcm9wZG93bnMuXG4gICAgICAgICAgICBjb25zdCAkZHJvcGRvd25zID0gJHRoaXNcbiAgICAgICAgICAgICAgICAuZmluZChzZWxlY3RvcnMuaW5wdXRzLmRyb3Bkb3duKVxuICAgICAgICAgICAgICAgIFt0aHVtYm5haWxJbWFnZXNPbmx5ID8gJ2ZpbHRlcicgOiAnbm90J10oc2VsZWN0b3JzLmlucHV0cy50aHVtYm5haWxJbWFnZURyb3Bkb3duKTtcblxuICAgICAgICAgICAgLy8gUmVtb3ZlIGltYWdlIG9wdGlvbiBmcm9tIGVhY2ggZHJvcGRvd24uXG4gICAgICAgICAgICAkZHJvcGRvd25zLmVhY2goKGluZGV4LCBlbGVtZW50KSA9PiB7XG4gICAgICAgICAgICAgICAgLy8gRHJvcGRvd24gZWxlbWVudC5cbiAgICAgICAgICAgICAgICBjb25zdCAkZHJvcGRvd24gPSAkKGVsZW1lbnQpO1xuXG4gICAgICAgICAgICAgICAgLy8gUmVtb3ZlIG9wdGlvbi5cbiAgICAgICAgICAgICAgICAkZHJvcGRvd24uZmluZChgW3ZhbHVlPVwiJHtmaWxlTmFtZX1cIl1gKS5yZW1vdmUoKTtcblxuICAgICAgICAgICAgICAgIC8vIFNldCB0byBkZWZhdWx0IHZhbHVlIGlmIHRoZXJlIGFyZSBubyBpbWFnZSBmaWxlIG9wdGlvbnMuXG4gICAgICAgICAgICAgICAgaWYgKCRkcm9wZG93bi5jaGlsZHJlbigpLmxlbmd0aCA8PSAxKSB7XG4gICAgICAgICAgICAgICAgICAgICRkcm9wZG93bi52YWwoJycpO1xuICAgICAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgICAgIC8vIFRyaWdnZXIgY2hhbmdlLlxuICAgICAgICAgICAgICAgICRkcm9wZG93bi50cmlnZ2VyKCdjaGFuZ2UnKTtcbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9XG5cbiAgICAgICAgLyoqXG4gICAgICAgICAqIERpc2FibGVzIG9yIGVuYWJsZXMgdGhlIHN1Ym1pdCBidXR0b25zLlxuICAgICAgICAgKi9cbiAgICAgICAgZnVuY3Rpb24gdG9nZ2xlU3VibWl0QnV0dG9ucygpIHtcbiAgICAgICAgICAgIC8vIEVuYWJsZSB0aGUgc3VibWl0IGJ1dHRvbnM/XG4gICAgICAgICAgICBsZXQgZG9FbmFibGVTdWJtaXRCdXR0b25zID0gdHJ1ZTtcblxuICAgICAgICAgICAgLy8gU2xpZGVzLlxuICAgICAgICAgICAgY29uc3QgJHNsaWRlcyA9ICR0aGlzLmZpbmQoc2VsZWN0b3JzLnNsaWRlUGFuZWwpO1xuXG4gICAgICAgICAgICAvLyBEaXNhYmxlIHN1Ym1pdCBidXR0b25zLCBpZiB0aGVyZSBhcmUgbm8gc2xpZGVzLlxuICAgICAgICAgICAgaWYgKCEkc2xpZGVzLmxlbmd0aCkge1xuICAgICAgICAgICAgICAgIGRvRW5hYmxlU3VibWl0QnV0dG9ucyA9IGZhbHNlO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICAvLyBEaXNhYmxlL0VuYWJsZSBzdWJtaXQgYnV0dG9ucy5cbiAgICAgICAgICAgIGVsZW1lbnRzLmJ1dHRvbnMuc3VibWl0XG4gICAgICAgICAgICAgICAgLmNoaWxkcmVuKClcbiAgICAgICAgICAgICAgICAubm90KCd1bCcpXG4gICAgICAgICAgICAgICAgLnByb3AoJ2Rpc2FibGVkJywgIWRvRW5hYmxlU3VibWl0QnV0dG9ucyk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICAgICAgICAvLyBJTklUSUFMSVpBVElPTlxuICAgICAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuXG4gICAgICAgIG1vZHVsZS5pbml0ID0gZG9uZSA9PiB7XG4gICAgICAgICAgICAvLyBBdHRhY2ggY2xpY2sgZXZlbnQgaGFuZGxlciB0byBzb3J0IGJ1dHRvbi5cbiAgICAgICAgICAgIGVsZW1lbnRzLmJ1dHRvbnMuc29ydC5vbignY2xpY2snLCBfb25Tb3J0QnV0dG9uQ2xpY2spO1xuXG4gICAgICAgICAgICAvLyBBdHRhY2ggZXZlbnQgaGFuZGxlcnMgdG8gc29ydCBhY3Rpb25zLCBzbGlkZSBwYW5lbCBkZWxldGUgYnV0dG9uIGFuZCBpbnB1dHMgZmllbGRzLlxuICAgICAgICAgICAgJHRoaXNcbiAgICAgICAgICAgICAgICAub24oJ3NvcnRzdGFydCcsIF9vblNvcnRTdGFydClcbiAgICAgICAgICAgICAgICAub24oJ3NvcnRzdG9wJywgX29uU29ydFN0b3ApXG4gICAgICAgICAgICAgICAgLm9uKCdjbGljaycsIHNlbGVjdG9ycy5pY29ucy5kZWxldGUsIF9vbkRlbGV0ZUljb25DbGljaylcbiAgICAgICAgICAgICAgICAub24oJ2tleXVwJywgc2VsZWN0b3JzLmlucHV0cy50aXRsZSwgX29uVGl0bGVLZXl1cClcbiAgICAgICAgICAgICAgICAub24oJ2NoYW5nZScsIHNlbGVjdG9ycy5pbnB1dHMuZmlsZSwgX29uSW1hZ2VBZGQpXG4gICAgICAgICAgICAgICAgLm9uKCdjbGljaycsIHNlbGVjdG9ycy5pY29ucy51cGxvYWQsIF9vblVwbG9hZEljb25DbGljaylcbiAgICAgICAgICAgICAgICAub24oJ2NsaWNrJywgc2VsZWN0b3JzLmljb25zLmltYWdlRGVsZXRlLCBfb25JbWFnZURlbGV0ZSlcbiAgICAgICAgICAgICAgICAub24oJ2NsaWNrJywgc2VsZWN0b3JzLmljb25zLmltYWdlTWFwLCBfb25JbWFnZU1hcClcbiAgICAgICAgICAgICAgICAub24oJ2NoYW5nZScsIHNlbGVjdG9ycy5pbnB1dHMuZHJvcGRvd24sIF9vbkltYWdlQ2hhbmdlKVxuICAgICAgICAgICAgICAgIC5vbignbW91c2VlbnRlcicsIHNlbGVjdG9ycy5jb25maWd1cmF0aW9uUm93LCBfb25Db25maWdSb3dNb3VzZUVudGVyKTtcblxuICAgICAgICAgICAgLy8gQXR0YWNoIGV2ZW50IGxpc3RlbmVycyB0byBzdWJtaXQgYnV0dG9ucy5cbiAgICAgICAgICAgIGVsZW1lbnRzLmJ1dHRvbnMuc3VibWl0U2F2ZS5vbignY2xpY2snLCBfb25TdWJtaXRTYXZlKTtcbiAgICAgICAgICAgIGVsZW1lbnRzLmJ1dHRvbnMuc3VibWl0UmVmcmVzaC5vbignY2xpY2snLCBfb25TdWJtaXRSZWZyZXNoKTtcblxuICAgICAgICAgICAgLy8gQXR0YWNoIGNsaWNrIGV2ZW50IGhhbmRsZXIgdG8gY3JlYXRlIGJ1dHRvbi5cbiAgICAgICAgICAgIGVsZW1lbnRzLmJ1dHRvbnMuY3JlYXRlLm9uKCdjbGljaycsIF9vbkNyZWF0ZUJ1dHRvbkNsaWNrKTtcblxuICAgICAgICAgICAgLy8gQWN0aXZhdGUgZmlyc3QgdGFiLlxuICAgICAgICAgICAgZWxlbWVudHMudGFiSGVhZGVyXG4gICAgICAgICAgICAgICAgLmNoaWxkcmVuKClcbiAgICAgICAgICAgICAgICAuZmlyc3QoKVxuICAgICAgICAgICAgICAgIC5hZGRDbGFzcygnYWN0aXZlJyk7XG5cbiAgICAgICAgICAgIC8vIEFjdGl2YXRlIGZpcnN0IHRhYiBjb250ZW50LlxuICAgICAgICAgICAgJHRoaXNcbiAgICAgICAgICAgICAgICAuZmluZChzZWxlY3RvcnMudGFiQm9keSlcbiAgICAgICAgICAgICAgICAuZmlyc3QoKVxuICAgICAgICAgICAgICAgIC5hZGRDbGFzcygnYWN0aXZlIGluJyk7XG5cbiAgICAgICAgICAgIC8vIFRyaWdnZXIgZHJvcGRvd24gY2hhbmdlIGV2ZW50IHRvIGhpZGUgdGhlIHJlbW92ZSBpY29uLCBpZiAnZG8gbm90IHVzZScgaXMgc2VsZWN0ZWQuXG4gICAgICAgICAgICAkdGhpc1xuICAgICAgICAgICAgICAgIC5maW5kKHNlbGVjdG9ycy5pbnB1dHMuZHJvcGRvd24pXG4gICAgICAgICAgICAgICAgLnRyaWdnZXIoJ2NoYW5nZScsIFtmYWxzZV0pO1xuXG4gICAgICAgICAgICAvLyBGaW5pc2ggaW5pdGlhbGl6YXRpb24uXG4gICAgICAgICAgICBkb25lKCk7XG4gICAgICAgIH07XG5cbiAgICAgICAgcmV0dXJuIG1vZHVsZTtcbiAgICB9XG4pO1xuIl19
